# (c) cavaliba.conf - app_sirene - views.py

import yaml
import json
from datetime import datetime
from datetime import timedelta

from django.utils import timezone
from django.shortcuts import render, redirect
from django.contrib import messages
from django.utils.translation import gettext as _
from django.views.decorators.cache import cache_page

import app_home.cache as cache
from app_home.configuration import get_configuration
from app_home.log import log, DEBUG, INFO, WARNING, ERROR, CRITICAL
from app_user.aaa import start_view
from app_user.aaa import get_aaa

from app_sirene.notify import aaa_message_allowed
from app_sirene.notify import get_public_default
from app_sirene.notify import get_public_active
from app_sirene.notify import get_message_enabled
from app_sirene.notify import get_message_by_id
from app_sirene.notify import archive_all
from app_sirene.notify import archive_expired

# create_notification in data/views HOOK
from app_sirene.notify import create_update
from app_sirene.notify import get_updates
from app_sirene.notify import reopen_message

from app_data.data import Instance

from app_sirene.forms import MessageUpdateForm

  

#-----------------------------------------
# public
#-----------------------------------------

# cache for X sec
#@cache_page(2)
def index(request):

    context = start_view(request, app="sirene", view="public") 
    if context["redirect"]:
        return redirect(context["redirect"])

    max_display = int(get_configuration("sirene", "PUBLIC_MAX_ITEMS"))
    #sort_order = get_configuration("sirene", "PUBLIC_SORT_ORDER")

      # skip public page, if 
    # - trusted ip / or authenticated
    # - configuration requires skip
    # - no toggle in GET URL
    skip_public = get_configuration("sirene", "PUBLIC_SKIP_TO_TRUSTED")
    keep_public = request.GET.get('public', 'no')

    if keep_public != "yes":
        if skip_public == "yes":
            aaa = get_aaa(request)
            if (aaa['trusted_ip'] or aaa['is_authenticated']):
                return redirect("app_sirene:anonymous")

    instances = get_public_active()

    if len(instances) == 0:
        instances = get_public_default()

    messages = []

    for instance in instances:

        if not instance.is_enabled:
            continue

        # no private (restricted) message for Anonymous
        if instance.is_field_true('is_restricted'):
            continue

        # full structure with datapoints
        ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)
        

        # extract relevant value
        message = {}
        try:
            message['title']       = ui_dict['displayname']
            message['severity']    = ui_dict['severity']['value']    # [{ value:'xxx', widget:'' }]
            message['bgcolor']     = ui_dict['bgcolor']['value']
            message['fgcolor']     = ui_dict['fgcolor']['value']
            message['body']        = ui_dict['content']['value']
            messages.append(message)
        except:
            continue

    context["pages"] = messages[:max_display]
    return render(request, 'app_sirene/public.html', context)



#-----------------------------------------
# anonymous
#-----------------------------------------
def anon_list(request):

    context = start_view(request, app="sirene", view="anonymous_list")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    
    if not (aaa["is_trusted_ip"] or aaa["is_authenticated"] or aaa["is_visitor"]):
        return redirect("app_sirene:index")

    instances = get_message_enabled()
    
    messages = []

    for instance in instances:

        # no private (restricted) message for Anonymous
        if instance.is_field_true('is_restricted'):
            continue

        # full structure with datapoints
        ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)
        

        # extract relevant value from (flat) Datapoints
        message = {}
        try:
            message['id'] = ui_dict['id']
        except:
            pass
        try:
            message['displayname'] = ui_dict['displayname']
            message['category'] = ui_dict['category']['value']    # [{ value:'xxx', widget:'' }]
            message['severity'] = ui_dict['severity']['value']    # [{ value:'xxx', widget:'' }]
        except:
            continue

        try:
            message['created_at'] = ui_dict['created_at']['value']
        except:
            message['created_at'] = ''
        try:
            message['updated_at'] = ui_dict['updated_at']['value']
        except:
            message['updated_at'] = ''

        messages.append(message)

    log(DEBUG, aaa=aaa, app="sirene", view="anonymous_list", action="get", status="OK", data="")

    context["messages"] = messages
    return render(request, 'app_sirene/anonymous.html', context)



def anon_detail(request, id=None):
    ''' display private message details for anonymous users'''

    context = start_view(request, app="sirene", view="anonymous_detail")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    aaa = get_aaa(request)
    if not (aaa['is_trusted_ip'] or aaa['is_authenticated']  or aaa["is_visitor"]):
        log(DEBUG, aaa=aaa, app="sirene", view="anonymous_detail", action="get", status="KO", data="access denied")
        return redirect("app_sirene:index")

    if not id:
        log(DEBUG, aaa=aaa, app="sirene", view="anonymous_detail", action="get", status="KO", data="no message")
        return redirect("app_sirene:anonymous")       


    instance = get_message_by_id(id=id)

    if not instance:
        log(DEBUG, aaa=aaa, app="sirene", view="anonymous_detail", action="get", status="KO", data="no message")
        return redirect("app_sirene:anonymous")       

    # exclude private (restricted) message from anonymous
    if instance.is_field_true('is_restricted'):
        return redirect("app_sirene:anonymous")

    # acive message only for anonymous
    if not instance.is_enabled:
        return redirect("app_sirene:anonymous")

    ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)


    # extract relevant value from (flat) Datapoints
    message = {}
    message['id']          = ui_dict['id']

    try:
        message['title']       = ui_dict['displayname']
        message['body']        = ui_dict['content']['value']
        message['category']    = ui_dict['category']['value']    # [{ value:'xxx', widget:'' }]
        message['severity']    = ui_dict['severity']['value']    # [{ value:'xxx', widget:'' }]
        message['bgcolor']     = ui_dict['severity__bgcolor']['value']
    except:
        return redirect("app_sirene:anonymous")

    try:
        message['created_at']  = ui_dict['created_at']['value']
    except:
        message['created_at']  = ''
    try:
        message['created_by']  = ui_dict['created_by']['value'][0]
    except:
        message['created_by']  = ''
    try:
        message['updated_at']  = ui_dict['updated_at']['value']
    except:
        message['updated_at']  = ''


    log(DEBUG, aaa=aaa, app="sirene", view="anonymous_detail", action="get", status="OK", 
        data=f"{message['title']}")


    context["message"] = message
    return render(request, 'app_sirene/anonymous_detail.html', context)


# -------------------------------------------
# private message list : name = 'private'
# -------------------------------------------
def private_list(request):

    context = start_view(request, app="sirene", view="message_list", 
        noauth="app_sirene:index", perm="p_sirene_access", noauthz="app_home:private",
        # explicitly allow visitor
        visitor = True  
        )
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]


    instances = get_message_enabled()
    pages = []

    for instance in instances:

        # SireneUser ORM object
        r = aaa_message_allowed(instance=instance, aaa=aaa)
        if not r:
            continue

        # full structure with datapoints
        ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)        


        # extract relevant value from (flat) Datapoints
        page = {}

        page['id'] = ui_dict['id']
        page['displayname'] = ui_dict['displayname']

        try:
            page['category'] = ui_dict['category']['value'][0]['value']    # [{ value:'xxx', widget:'' }]
        except:
            continue      

        try:
            page['severity'] = ui_dict['severity']['value'][0]['value']    # [{ value:'xxx', widget:'' }]
        except Exception as e:
            print(e)
            continue

        try:            
            page['bgcolor'] = ui_dict['severity__bgcolor']['value']
        except:
            continue


        try:
            page['created_at'] = ui_dict['created_at']['value']
        except:
            continue
        try:
            page['updated_at'] = ui_dict['updated_at']['value']
        except:
            page['updated_at'] = ''

        if instance.is_field_true('is_restricted'):
            page['is_restricted'] = True
        else:
            page['is_restricted'] = False



        pages.append(page)
   
    
    log(DEBUG, aaa=aaa, app="sirene", view="message", action="list", status="OK", data="")

    context["pages"] = pages
    return render(request, 'app_sirene/private.html', context)





# -------------------------------------------
# private message detail : name = 'detail'
# -------------------------------------------

def private_detail(request, id=None):
    ''' GET id - display details + admin tools'''

    context = start_view(request, app="sirene", view="detail", 
        noauth="app_sirene:index", perm="p_sirene_detail", noauthz="app_sirene:private",
        visitor=True)
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    archive_expired(aaa=aaa)

    if not id:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="message", action="detail", status="KO", data=f"no ID")
        return redirect("app_sirene:private")

    instance = get_message_by_id(id=id)
    
    if not instance:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="message", action="detail", status="KO", data=f"Not found {id}")
        return redirect("app_sirene:private")

    # SireneUser ORM object
    r = aaa_message_allowed(instance=instance, aaa=aaa)
    if not r:
        return redirect("app_sirene:private")


    ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)

    #instance.print()

    # extract relevant value from (flat) Datapoints
    page = {}
    page['id'] = instance.id
    page['is_enabled'] = instance.is_enabled

    try:
        page['title']       = ui_dict['displayname']
    except:
        page['title']       = ""
    
    try:        
        page['body']        = ui_dict['content']['value']
    except:
        page['body']        = ""
    
    try:
        page['category']    = ui_dict['category']['value'][0]['value']
    except:
        page['category']    = ""
    
    try:
        page['severity']    = ui_dict['severity']['value'][0]['value']
    except:
        page['severity']    = ""
    try:
        page['bgcolor']     = ui_dict['severity__bgcolor']['value']
    except:
        page['bgcolor']     = "success"
    
    try:
        page['created_at']  = ui_dict['created_at']['value']
    except:
        page['created_at']  = ''

    try:
        page['created_by']  = ui_dict['created_by']['value'][0]['key']
    except:
        page['created_by']  = ''

    try:
        page['updated_at']  = ui_dict['updated_at']['value']
    except:
        page['updated_at']  = ''

    try:
        page['updated_by']  = ui_dict['updated_by']['value'][0]['key']
    except:
        page['updated_by']  = ''


    try:
        page['removed_at']  = ui_dict['removed_at']['value']
    except:
        page['removed_at']  = ''

    try:
        page['removed_by']  = ui_dict['removed_by']['value'][0]['key']
    except:
        page['removed_by']  = ''


    if instance.is_field_true('is_restricted'):
        page['is_restricted'] = True
    else:
        page['is_restricted'] = False

    try:
        page['publicpage'] = ui_dict['public_page']['value'][0]['display']
    except:
        page['publicpage'] = ""

    try:
        page['email_count'] = ui_dict['email_count']['value']
        page['sms_count'] = ui_dict['sms_count']['value']
    except:
        page['email_count'] = 0
        page['sms_count'] = 0

    try:
        page['notified_username'] = ', '.join(instance.fields['notified_username'].value)
        page['notified_username_count'] = len(instance.fields['notified_username'].value)
    except:
        pass

    # notify_* fields:
    #    [{'key': 'testuser01', 'display': 'Test User 01'}]
    #    ['testgroup02']
    #    [{'key': 'testapp01', 'display': 'Test APP 01', 'id': id}]
    page['notified_targets'] = []
    for fieldname in ui_dict:
        if fieldname.startswith('notify_'):
            for target in ui_dict[fieldname]['value']:
                if type(target) is dict:
                    v = target.get('display', target.get('key', '?'))
                elif type(target) is list:
                    v = target[0]
                elif type(target) is str:
                    v = target
                page['notified_targets'].append(v)

    try:
        page['template'] = ui_dict['template']['value']
    except:
        page['template'] = "N/A"


    log(DEBUG, aaa=aaa, app="sirene", view="message", action="detail", status="OK", data=f"{page['title']}")


    # updates
    
    a = get_updates(instance=instance)[::-1]
    if len(a) > 0:
        context['updates'] = a



    context["page"] = page
    return render(request, 'app_sirene/detail.html', context)




#-----------------------------------------
# name = update  (+archive)
#-----------------------------------------
def update(request, id=None):

    ''' Edit MessageUpdateForm - GET/POST'''

    context = start_view(request, app="sirene", view="update", 
        noauth="app_sirene:index", perm="p_sirene_update", noauthz="app_sirene:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]


    if not id:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="update", action="id", status="KO", data=f"No ID")
        return redirect("app_sirene:private")

    instance = get_message_by_id(id)

    if not instance:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="update", action="instance", status="KO", data=f"{id} not found")
        return redirect("app_sirene:private")
    

    # SireneUser ORM object
    r = aaa_message_allowed(instance=instance, aaa=aaa)
    if not r:
        return redirect("app_sirene:private")

    if request.method == "POST":
        form = MessageUpdateForm(request.POST)

        if form.is_valid():
            cd = form.cleaned_data
            mu = {}
            mu['content']    = cd["content"]
            mu['has_email']  = cd['has_email']
            mu['has_sms']    = cd['has_sms']
            mu['archive']    = cd['archive']
            
            action = 'update'
            if mu['archive']:
                action="archive"

            err, email_count, sms_count = create_update(instance=instance, aaa=aaa, mu=mu)
            if err:
                messages.add_message(request, messages.ERROR, _("Failed"))
                log(WARNING, aaa=aaa, app="sirene", view="update", action=action, status="KO", 
                    data=f"{instance.id} - {instance.displayname} - {err}")
                
            else:
                messages.add_message(request, messages.SUCCESS, f"OK - {email_count} emails / {sms_count} SMS")
                log(INFO, aaa=aaa, app="sirene", view="update", action=action, status="OK", 
                    data=f"{instance.id} - {instance.displayname} - {email_count} emails - {sms_count} sms")
                return redirect("app_sirene:detail", instance.id)

        else:
            messages.add_message(request, messages.ERROR, _("Invalid form"))
            log(ERROR, aaa=aaa, app="sirene", view="update", action="valid", status="KO", 
                data=f"invalid form - {instance.id} - {instance.displayname}")


    else:
        form = MessageUpdateForm()
    
    
    log(INFO, aaa=aaa, app="sirene", view="update", action="edit", status="OK", 
        data=f"{instance.id} {instance.displayname}")


    # extract relevant value from (flat) Datapoints
    message = {}
    message['id'] = instance.id

    context["message"] = message
    context["form"] = form
    return render(request, 'app_sirene/update.html', context)



#-----------------------------------------
# name = reopen
#-----------------------------------------
def reopen(request, id=None):

    ''' Reopen archived message'''

    context = start_view(request, app="sirene", view="reopen", 
        noauth="app_sirene:index", perm="p_sirene_update", noauthz="app_sirene:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]


    if not id:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="reopen", action="id", status="KO", data=f"No ID")
        return redirect("app_sirene:private")

    instance = get_message_by_id(id)
    if not instance:
        messages.add_message(request, messages.ERROR, _("Not found"))
        log(ERROR, aaa=aaa, app="sirene", view="reopen", action="instance", status="KO", data=f"{id} not found")
        return redirect("app_sirene:private")
    

    # SireneUser ORM object
    r = aaa_message_allowed(instance=instance, aaa=aaa)
    if not r:
        return redirect("app_sirene:private")

    result = reopen_message(instance=instance)
    if not result:
        messages.add_message(request, messages.ERROR, _("Failed"))
        log(WARNING, aaa=aaa, app="sirene", view="reopen", action="reopen", status="KO", 
            data=f"{instance.id} - {instance.displayname}")
                
    else:
        messages.add_message(request, messages.SUCCESS, _("Reopened"))
        log(INFO, aaa=aaa, app="sirene", view="reopen", action="reopen", status="OK", 
            data=f"{instance.id} - {instance.displayname}")
    
    
    return redirect("app_sirene:detail", instance.id)





#-----------------------------------------
# flushall : name = flushall
#-----------------------------------------
def flushall_view(request):
    ''' POST with CSRF  > remove all is_visble messages'''

    context = start_view(request, app="sirene", view="config_flushall", 
        noauth="app_sirene:index", perm="p_sirene_flushall", noauthz="app_sirene:index")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]


    if request.method != 'POST':
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="sirene", view="flushall", action="check", status="KO", data="method not supported")
        return redirect("app_sirene:index")


    count = archive_all(aaa=aaa)

    log(INFO, aaa=aaa, app="sirene", view="flushall", action="post", status="OK", data=f"flushall {count} entries")
    messages.add_message(request, messages.SUCCESS, _("OK"))
    return redirect("app_sirene:private")





#-----------------------------------------
# name = archive
#-----------------------------------------
def archive(request):

    context = start_view(request, app="sirene", view="archive", 
        noauth="app_sirene:index", perm="p_sirene_history", noauthz="app_sirene:private",
        visitor=True)
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    pages = []

    for instance in Instance.iterate_classname(classname="sirene_message", first=0, last=500, enabled="no", expand=True):

        # permission
        if not aaa_message_allowed(instance=instance, aaa=aaa):
            continue

        # full structure with datapoints
        ui_dict = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False, skip_enumerate=False)        


        # extract relevant value from (flat) Datapoints
        page = {}
        page['id'] = ui_dict['id']
        try:
            page['displayname'] = ui_dict['displayname']
        except:
            page['displayname'] = ""

        try:
            page['category'] = ui_dict['category']['value'][0]['value']    # [{ value:'xxx', widget:'' }]
        except:
            page['category'] = ""
        try:
            page['severity'] = ui_dict['severity']['value'][0]['value']    # [{ value:'xxx', widget:'' }]
        except:
            page['severity'] = ""
        try:
            page['bgcolor']  = ui_dict['severity__bgcolor']['value']
        except Exception as e:
            page['bgcolor']  = ""

        # no update button / no send to archive
        page['is_enabled'] = False


        if instance.is_field_true('is_restricted'):
            page['is_restricted'] = True
        else:
            page['is_restricted'] = False


        try:
            page['publicpage'] = ui_dict['public_page']['value'][0]['display']
        except:
            page['publicpage'] = ""
            
        try:
            page['created_at']  = ui_dict['created_at']['value']
        except:
            page['created_at']  = ''

        try:
            page['created_by']  = ui_dict['created_by']['value'][0]['key']
        except:
            page['created_by']  = ''

        try:
            page['updated_at']  = ui_dict['updated_at']['value']
        except:
            page['updated_at']  = ''

        try:
            page['updated_by']  = ui_dict['updateed_by']['value'][0]['key']
        except:
            page['updated_by']  = ''


        try:
            page['removed_at']  = ui_dict['removed_at']['value']
        except:
            page['removed_at']  = ''

        try:
            page['removed_by']  = ui_dict['removed_by']['value'][0]['key']
        except:
            page['removed_by']  = ''    


        pages.append(page)



    log(DEBUG, aaa=aaa, app="sirene", view="archive", action="list", status="OK", data="")

    context["pages"] = pages
    return render(request, 'app_sirene/archive.html', context)

