# (c) cavaliba.com - test_schema

import yaml

from django.test import override_settings
from django.test import TestCase
from django.test import TransactionTestCase
from django.urls import reverse
from django.core.cache import cache as cache_django
from django.core.management import call_command

from tests import helper

import app_home.cache as cache
from app_data.models import DataInstance
from app_data.data import Instance
from app_data.loader import load_broker

from app_data.schema import Schema


class TestDataSchema(TestCase):

    # fixtures = ["test"]

    def setUp(self):
        cache.clear()

    def test_empty_instance(self):
        i = Instance()
        self.assertIsNotNone(i)

    def test_create_schema(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  icon: testicon
                  unittest_option: edit
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertEqual(schema.displayname, 'Unittest Schema')
        self.assertEqual(schema.icon, 'testicon')
        self.assertIn('unittest_option', schema.options)

    def test_listsall_schema(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  unittest_option: edit
                  icon: testicon
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schemas = Schema.listall()
        self.assertIs(len(schemas), 1)


class TestDataSchemaOptions(TestCase):

    # fixtures = ["test"]

    def setUp(self):
        cache.clear()

    # keyname_mode
    # ------------

    def test_keyname_mode_edit(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  icon: testicon
                  keyname_mode: edit
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertIn('keyname_mode', schema.options)
        self.assertEqual(schema.keyname_mode, 'edit')

    def test_keyname_mode_auto(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  icon: testicon
                  keyname_mode: auto
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertIn('keyname_mode', schema.options)
        self.assertEqual(schema.keyname_mode, 'auto')

    def test_keyname_mode_default(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertEqual(schema.keyname_mode, 'edit')

    def test_keyname_mode_invalid(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  icon: testicon
                  keyname_mode: invalid
            ''')
        aaa = {'perms': ['p_schema_create', 'p_schema_update']}
        _ = load_broker(datalist=datalist, aaa=aaa, verbose=True)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertEqual(schema.keyname_mode, 'edit')

    def test_keyname_mode_auto_result(self):
        datalist = yaml.safe_load('''
            - classname: _schema
              keyname: unittestschema
              displayname: 'Unittest Schema'
              _options:
                  icon: testicon
                  keyname_mode: auto

            - classname: unittestschema
              keyname: utest01

            - classname: unittestschema

            ''')

        aaa = {'perms': ['p_schema_create',
                         'p_schema_update',
                         'p_data_admin',
                         ]}

        _ = load_broker(datalist=datalist, aaa=aaa, verbose=False)

        schema = Schema.from_name('unittestschema')
        self.assertIsNotNone(schema)
        self.assertIn('keyname_mode', schema.options)
        self.assertEqual(schema.keyname_mode, 'auto')

        inst1 = Instance.from_keyname(classname='unittestschema', keyname='utest01')
        self.assertIsNone(inst1)

        qs = DataInstance.objects.filter(classname='unittestschema')
        self.assertEqual(len(qs), 2)

        for obj in qs:
            k = obj.keyname
            self.assertEqual(len(k), 36)

    # keyname_label
    # -------------

    # displayname_label
    # ------------------
