# (c) cavaliba.com - data - schema.py

import uuid
import copy
import yaml
import sys

from django.forms.models import model_to_dict


from app_home.cavaliba import TRUE_LIST
import app_home.cache as cache

from app_data.models import DataClass
from app_data.models import DataSchema


# New V3.19

class Schema:

    def __init__(self):

        self.classname = None
        self.displayname = None
        self.is_enabled = None

        self.options = {}
        # keyname_mode: edit(*) | auto
        # displayname_option
        # icon: XXXXXX

        ####self.icon = None
        self.order = None
        self.page = None

        # _sections

        # permissions
        self.p_admin = None
        self.p_create = None
        self.p_read = None
        self.p_update = None
        self.p_delete = None


        self.fields = {}
        self.ordered_fields = {}    # self.ordered_fields['page'][order] = fieldname

        # @properties 
        # X icon
        # X keyname_mode
        # X keyname_label




    # -----------
    # properties
    # -----------
    
    # keyname_mode: edit(*)|auto - V3.19
    # ----------------------------------
    @property
    def keyname_mode(self):
        v = self.options.get("keyname_mode","edit")
        if v not in ["edit", "auto"]:
            v = "edit"
        return v

    @keyname_mode.setter
    def keyname_mode(self, value):
        if value in ["edit", "auto"]:
            self.options["keyname_mode"] = value
        else:
            self.options["keyname_mode"] = "edit"


    @classmethod
    def create_keyname(self):
        return str(uuid.uuid4())



    # keyname_label - V3.20
    # ----------------------
    @property
    def keyname_label(self):
        return self.options.get("keyname_label", "Keyname")

    # displayname_label - V3.20
    # ---------------------------
    @property
    def displayname_label(self):
        return self.options.get("displayname_label", "Displayname")



    # icon - V3.20
    # ------------
    @property
    def icon(self):
        return self.options.get("icon", "fa-question")
            
    @icon.setter
    def icon(self, value):
        #self.icon = value
        self.options["icon"] = value


    # -------------------------------------------------------------------------

    def __str__(self):
        return f"{self.classname}"


    @classmethod
    def listall(cls):
        ''' returns a list[] of all DataClass as Schema() objects'''

        reply = []
        names = DataClass.objects.values_list("keyname", flat=True).all()
        for name in names:
            schema = cls.from_name(name)
            reply.append(schema)
        return reply


    @classmethod
    def displayname_dict(cls):
        ''' returns a dict of all classname => displayname '''
        
        # [  { keyname:displayname}, {}, ...]
        names_list = DataClass.objects.values("keyname", "displayname").all()
        
        # to pure dict : { keyname: displayname}
        names = {}
        for adict in names_list:
            names[ adict['keyname']] = adict['displayname']

        return names


    @classmethod
    def from_name(cls, classname=None):
        ''' load from DB or None'''

        if not classname:
            return
        
        if classname in cache.cache_schema2:
            return copy.deepcopy(cache.cache_schema2[classname])
            
        obj = DataClass.objects.filter(keyname=classname).first()
        if not obj:
            cache.cache_schema2.pop(classname, None)
            return


        schema = cls()
        schema.classname = classname
        schema.displayname = obj.displayname
        schema.is_enabled = obj.is_enabled

        # first level options (3.20) >> @property
        #schema.icon = obj.icon
        

        schema.page = obj.page
        schema.order = obj.order

        schema.p_admin = obj.p_admin
        schema.p_create = obj.p_create
        schema.p_read = obj.p_read
        schema.p_update = obj.p_update
        schema.p_delete = obj.p_delete

        # options
        try:
            schema.options = yaml.safe_load(obj.options)
            if type(schema.options) is not dict:
                schema.options = {}
        except:
            schema.options = {}


        db_fields = DataSchema.objects.filter(classname=classname).order_by("order")
        schema.fields = {}
        schema.ordered_fields = {}              # self.ordered_fields['page'][order] = [fieldname, ...]

        if db_fields:
            for field in db_fields:

                m = {}
                if field.displayname and len(field.displayname) >0:
                    m['displayname'] = field.displayname
                if len(field.description) > 0:
                    m['description'] = field.description
                m['dataformat'] = field.dataformat
                if field.dataformat_ext and len(field.dataformat_ext) > 0:
                    m['dataformat_ext'] = field.dataformat_ext
                m['order'] = field.order
                if field.page and len(field.page) > 0:
                    m['page'] = field.page

                if field.cardinal_min > 0:
                    m['cardinal_min'] = field.cardinal_min

                if field.cardinal_max != 1:
                    m['cardinal_max'] = field.cardinal_max
                if not field.is_enabled:
                    m['is_enabled'] = field.is_enabled
                if field.default_value and len(field.default_value) > 0:
                    m['default_value'] = field.default_value

                schema.fields[field.keyname] = m        

                order = field.order
                page = field.page
                if not page:
                    page = "Default"

                if page not in schema.ordered_fields:
                    schema.ordered_fields[page] = {}
                if order not in schema.ordered_fields[page]:
                    schema.ordered_fields[page][order]= []

                schema.ordered_fields[page][order].append(field.keyname)


        # cache update
        cache.cache_schema2[classname] = copy.deepcopy(schema)
        return schema
    
    


    @classmethod
    def has_schema_read_permission(cls, aaa=None):

        try:
            return 'p_schema_read' in aaa['perms']
        except:
            return False
        


    
    def to_yaml(self):
        ''' retrun as nice YAML text'''
        r = []

        r.append("- classname: _schema")

        head = {}
        head['keyname'] = self.classname
        if self.displayname and len(self.displayname) > 0:
            head['displayname'] = self.displayname
        if not self.is_enabled:
            head['is_enabled'] = False

        zz = yaml.dump(head, allow_unicode=True, sort_keys=False)
        for l in zz.splitlines():
            r.append("  " + l)

        
        r.append("")
        
        r.append("  _options:")
        options = {}
        if self.icon and len(self.icon) > 0:
            options['icon'] = self.icon
        options['keyname_mode'] = self.keyname_mode
        if self.keyname_label and len(self.keyname_label) > 0:
            options['keyname_label'] = self.keyname_label
        if self.displayname_label and len(self.displayname_label) > 0:
            options['displayname_label'] = self.displayname_label
        if self.p_admin and len(self.p_admin) > 0:
            options['p_admin'] = self.p_admin
        if self.p_create and len(self.p_create) > 0:
            options['p_create'] = self.p_create
        if self.p_read and len(self.p_read) > 0:
            options['p_read'] = self.p_read
        if self.p_update and len(self.p_update) > 0:
            options['p_update'] = self.p_update
        if self.p_delete and len(self.p_delete) > 0:
            options['p_delete'] = self.p_delete


        zz = yaml.dump(options, allow_unicode=True, sort_keys=False)
        for l in zz.splitlines():
            r.append("    " + l)



        current_page = None
        for page in self.ordered_fields:                # self.ordered_fields['page'][order] = [fieldname, ]
            for order in self.ordered_fields[page]:
                for keyname in self.ordered_fields[page][order]:
                    field = self.fields[keyname]

                    if page != current_page:
                        r.append("")
                        r.append("  # --- " + page)
                        current_page = page

                    r.append("")
                    r.append("  " + keyname + ':')

                    zz = yaml.dump(field, allow_unicode=True, sort_keys=False)
                    for l in zz.splitlines():
                        r.append("    " + l)

            
        return '\n'.join(r)



    def save(self):
        ''' save self to DataClass DB '''

        # exists ?
        if not self.classname:
            return
        
        obj = DataClass.objects.filter(keyname=self.classname).first()
        if not obj:
            obj = DataClass()
            obj.keyname = self.classname

        obj.displayname = self.displayname
        obj.is_enabled = self.is_enabled

        #obj.icon = self.icon
        obj.page = self.page
        obj.order = self.order


        myyaml = {}
        myyaml["keyname_mode"] = self.keyname_mode
        myyaml["keyname_label"] = self.keyname_label
        myyaml["displayname_label"] = self.displayname_label
        myyaml["icon"] = self.icon


        # NEXT : duplicate first level options (icon, p_*) in options field ?

        obj.options = yaml.dump(myyaml, allow_unicode=True, sort_keys=True)

        # permissions remain first level attributes
        # although under '_options:' in external YAML files
        obj.p_admin = self.p_admin
        obj.p_create = self.p_create
        obj.p_read = self.p_read
        obj.p_update = self.p_update
        obj.p_delete = self.p_delete

        obj.save()
