# (c) cavaliba.com - data - data.py

import json
import yaml
import copy
import uuid 
import sys

from pprint import pprint

from django.forms.models import model_to_dict
from django.utils.translation import gettext as _
from django.utils import timezone


from app_home.configuration import get_configuration
from app_home.cavaliba import TRUE_LIST
import app_home.cache as cache
from app_home.log import log, DEBUG, INFO, WARNING, ERROR, CRITICAL


from .fieldtypes.field_string        import FieldString
from .fieldtypes.field_int           import FieldInt
from .fieldtypes.field_boolean       import FieldBoolean
from .fieldtypes.field_schema        import FieldSchema
from .fieldtypes.field_group         import FieldGroup
from .fieldtypes.field_role          import FieldRole
from .fieldtypes.field_ipv4          import FieldIPV4
from .fieldtypes.field_float         import FieldFloat
from .fieldtypes.field_date          import FieldDate
from .fieldtypes.field_datetime      import FieldDatetime
from .fieldtypes.field_time          import FieldTime
from .fieldtypes.field_user          import FieldUser
from .fieldtypes.field_text          import FieldText
from .fieldtypes.field_enumerate     import FieldEnumerate
from .fieldtypes.field_external      import FieldExternal
from .fieldtypes.field_file          import FieldFile


from app_data.models import DataClass
from app_data.models import DataSchema
from app_data.models import DataInstance
from app_data.models import DataEAV

# V3.19 - Schema2
from app_data.schema import Schema


from .permissions import has_schema_delete_permission
from .permissions import has_schema_update_permission
from .permissions import has_schema_create_permission

from .permissions import has_schemafield_delete_permission
from .permissions import has_schemafield_update_permission
from .permissions import has_schemafield_create_permission

from .permissions import has_read_permission_on_class
from .permissions import has_edit_permission_on_class
from .permissions import has_delete_permission_on_class
from .permissions import has_create_permission_on_class

FIELD_CONSTRUCTOR_TABLE = { 
    "string":FieldString,
    "int":FieldInt,
    "float":FieldFloat,
    "boolean":FieldBoolean,
    "ipv4":FieldIPV4,
    "date":FieldDate,
    "datetime":FieldDatetime,
    "time":FieldTime,
    "text":FieldText,
    "user":FieldUser,
    "group":FieldGroup,
    "role":FieldRole,
    "schema": FieldSchema,
    "enumerate": FieldEnumerate,
    "external": FieldExternal,
    "file": FieldFile,
    }

# YAML export
class MyYamlDumper(yaml.SafeDumper):
    def write_line_break(self, data=None):
        super().write_line_break(data)
        if len(self.indents) < 2:
            super().write_line_break()


# -------------------------------------------------------------------------
# Task : bigset/count update
# -------------------------------------------------------------------------
def update_bigset():

    dataclasses = get_classes()

    bigset_size = int(get_configuration("data","DATA_BIGSET_SIZE"))

    for classobj in dataclasses:
        count = classobj.datainstance_set.count()
        classobj.count_estimation = count
        if count > bigset_size:
            classobj.is_bigset = True
            # but no switch back to false
        classobj.save()            
        print(f"update_bigset: class={classobj.keyname} count={classobj.count_estimation} bigset={classobj.is_bigset}")


def count_instance(classname=None):

    classobj = get_class_by_name(classname)
    count = classobj.datainstance_set.count()
    return count


# -------------------------------------------------------------------------
# Class Helpers
# -------------------------------------------------------------------------

def get_classes(is_enabled=None):

    if is_enabled:
        return DataClass.objects.order_by("order").filter(is_enabled=is_enabled)
    else:
        return DataClass.objects.order_by("order").all()



def get_class_by_name(keyname):
  
    if keyname not in cache.cache_classname:
        obj = DataClass.objects.filter(keyname=keyname).first()
        if obj:
            cache.cache_classname[keyname] = obj
            return obj
    else:
        return cache.cache_classname[keyname]

    # Non-cached version
    #return  DataClass.objects.filter(keyname=keyname).first()


def get_class_by_id(id):

    ''' 
    get DataClass DB object by pk=id 
    use cache
    '''

    if id not in cache.cache_classid:
        obj = DataClass.objects.filter(pk=id).first()
        if obj:
            cache.cache_classid[id] = obj
            return obj
    else:
        return cache.cache_classid[id]

    # Non-cached version
    # dataclass = DataClass.objects.filter(pk=id).first()
    # return dataclass


# -------------------------------------------------------------------------
# Instance Helpers
# -------------------------------------------------------------------------

def get_instances(classname = None, is_enabled=None):
    ''' => Queryset'''

    dataclass = get_class_by_name(classname)
    if is_enabled:
        instances = DataInstance.objects.filter(classobj=dataclass).select_related('classobj').filter(is_enabled=is_enabled)
    else:
        instances = DataInstance.objects.filter(classobj=dataclass).select_related('classobj').all()
    return instances



# fast query to select one field value 
# select keyname, fieldname from instance where classname = classname
# select keyname, *         from instance where classname = classname
def get_instances_raw_json(classname = None, is_enabled=True, fieldname = None):
    
    reply = {}

    classobj = get_class_by_name(classname)
    if is_enabled:
        instances = DataInstance.objects.filter(classobj=classobj).filter(is_enabled=is_enabled)
    else:
        instances = DataInstance.objects.filter(classobj=classobj)

    # extract json
    for iobj in instances:
        jsondata = json.loads(iobj.data_json)
        data = jsondata
        # filter on fieldname
        if fieldname:
            if fieldname in jsondata:
                data = jsondata[fieldname]
                reply[iobj.keyname] = data
    return reply



def get_instance_by_name(iname=None, classobj=None, classname=None):
    
    if not classobj:
        classobj = get_class_by_name(classname)
    if not classobj:
        return
    if not iname:
        return
    
    obj = cache.instance_get(classobj.keyname, iname)
    if not obj:
        obj = DataInstance.objects.filter(classobj=classobj, keyname=iname).select_related('classobj').first()
        cache.instance_store(classobj.keyname, iname, obj)
    return obj
    





def get_db_instance_by_id(id=None):
    ''' 3.21.0 '''

    if not id:
        return

    try:
        return DataInstance.objects.get(pk=id)
    except DataInstance.DoesNotExist:
        return



# -------------------------------------------------------------------------
# Schema
# -------------------------------------------------------------------------

def get_schema(classname=None):

    # { 
    # 'fieldname':{
    #         'displayname':''
    #         'description':''
    #         'dataformat':''
    #         'dataformat_ext':''
    #         'order':
    #         'cardinal_min':
    #         'cardinal_max':
    #         'is_multi': True/False
    #         'default_value':
    #         'is_injected': False
    #  }, 
    #  ...
    # }
    
    if not classname:
        return {}

    reply_dict = {}

    classobj = get_class_by_name(classname)
    if not classobj:
        return {}

    # cache hit ?
    if classname in cache.cache_schema:
        return cache.cache_schema[classname]

    db_schema = DataSchema.objects.filter(classname=classname).order_by("order")

    if not db_schema:
        return {}

    # convert to schema_dict !
    dict_attributs =  [ "displayname", "description", "dataformat", "dataformat_ext", "order", "page",
        "cardinal_min","cardinal_max", "default_value"  ]     

    for entry in db_schema:
        m = model_to_dict(entry, fields=dict_attributs)

        # by default schema fields are not injected at schema creation time
        m['is_injected'] = False

        reply_dict[entry.keyname] = m

    # cache update
    cache.cache_schema[classobj.keyname] = copy.deepcopy(reply_dict)
    return reply_dict


# --------------------------------------------------------
# INSTANCE
# --------------------------------------------------------
class Instance:
        
    def __init__(self, 
                 iobj=None, iname=None,
                 classname=None, 
                 classobj=None,
                 schema=None,
                 expand=False):

        self.classname = None
        self.classobj = None
        self.schema2 = None

        self.keyname = None
        self.iobj = None

        self.is_enabled = None     # True / False
        self.displayname = None

        self.p_read = None
        self.p_update = None
        self.p_delete = None

        self.json = None
        self.value = []
        self.fields = {}    # key = field keyname

        self.errors = []
        self.last_update = None


        if classname:
            self.classname = classname

        if classobj:
            self.classobj = classobj
            self.classname = self.classobj.keyname


        # check  iobj is a DataInstance
        if iobj:
            if not type(iobj) is DataInstance:
                return

        # find existing by  iname
        if not iobj:
            if classobj:
                if iname:
                    iobj = get_instance_by_name(iname=iname, classobj=classobj)
            elif self.classname:
                if iname:
                    iobj = get_instance_by_name(iname=iname, classname=self.classname)

        # if found in DB
        if iobj:
            self.iobj = iobj
            self.keyname = iobj.keyname
            self.is_enabled = iobj.is_enabled
            self.displayname = iobj.displayname
            self.p_read = iobj.p_read
            self.p_update = iobj.p_update
            self.p_delete = iobj.p_delete
            self.last_update = iobj.last_update
            if not self.classobj:
                self.classobj = iobj.classobj
            if not self.classname:
                self.classname = self.classobj.keyname
                        
        # new emptyInstance : iobj is None
        else:
            if iname:
                self.keyname = iname
            self.is_enabled = True
            self.displayname = None

            # set classobj from classname
            if not self.classobj:
                self.classobj = get_class_by_name(self.classname)
            if self.classobj:
                if not self.classobj.is_enabled:
                    self.classobj = None

        if not schema:
            schema = get_schema(classname=self.classname)

        # unpack existing iobj JSON structure     
        if iobj:
            try:
                self.json = json.loads(iobj.data_json)      
            except:
                self.json = None

        # create fields, and fill with json if available
        for fieldname, fieldschema in schema.items():
            # safety: don't inject a bad external keyname (missing prefix _)
            if fieldschema["dataformat"] == "external":
                if not fieldname.startswith('_'):
                    continue
            constructor = FIELD_CONSTRUCTOR_TABLE.get(fieldschema["dataformat"], FieldString)
            self.fields[fieldname] = constructor(fieldname, fieldschema, self.json)


        # V3.19+ ; add/unpack Schema2 class, with options
        if self.classname:
            self.schema2 = Schema.from_name(self.classname)


        # expand injected fields
        if expand:    
            self.expand_injected()

        self.set_options()



    # new 3.19 - specific constructors
    @classmethod
    def new_from_names(cls, classname=None, keyname=None):
        ''' NEW / Empty '''
        instance = cls(classname=classname, iname=keyname)
        return instance

    @classmethod
    def load_from_names(cls, classname=None, keyname=None, expand=False):
        ''' load from DB'''
        instance = cls(classname=classname, iname=keyname, expand=expand)
        if instance.is_bound:
            return instance

    @classmethod
    def load_from_iobj(cls, iobj=None, expand=False):
        ''' load from iob object (DB object)'''
        if not iobj:
            return
        instance = cls(classname=iobj.classname, iobj=iobj, expand=expand)
        if instance.is_bound:
            return instance

    @classmethod
    def from_names(cls, classname=None, keyname=None, expand=False):
        ''' load from DB or create'''
        instance = cls(classname=classname, iname=keyname, expand=expand)
        return instance

    @classmethod
    def from_classname(cls, classname=None):
        ''' Empty from classname'''
        instance = cls(classname=classname)
        return instance

    @classmethod
    def from_id(cls, id=None, expand=False):
        ''' existing or None - v3.21.0'''
        iobj = get_db_instance_by_id(id)
        if not iobj:
            return
        instance = cls(classname=iobj.classname, iobj=iobj, expand=expand)
        if instance.is_bound:
            return instance

    # NEXT: iterator
    @classmethod
    def iterate_classname(cls, classname=None, first=None, last=None, enabled=None, expand=False):
        ''' returns a list[] of Instance()'''
        
        if not classname:
            return []
        
        instances = []

        if type(first) is int and type(last) is int:
            if enabled == "yes":
                instances = DataInstance.objects.filter(classname=classname, is_enabled=True)[first:last]
            elif enabled == "no":
                instances = DataInstance.objects.filter(classname=classname, is_enabled=False)[first:last]
            else:
                instances = DataInstance.objects.filter(classname=classname)[first:last]
        else:
            if enabled == "yes":
                instances = DataInstance.objects.filter(classname=classname, is_enabled=True)
            elif enabled == "no":
                instances = DataInstance.objects.filter(classname=classname, is_enabled=False)
            else:
                instances = DataInstance.objects.filter(classname=classname)
        
        reply = []

        
        for iobj in instances:
            instance = cls.load_from_iobj(iobj=iobj, expand=expand)
            if instance:
                reply.append(instance)

        return reply



    def __str__(self):
        return f"{self.classname}:{self.keyname}"


    def print(self):
   
        print(f"{self.classname}:{self.keyname}")
        print(f"    id:          {self.id}")
        print(f"    is_enabled:  {self.is_enabled}")
        print(f"    displayname: {self.displayname}")
        print(f"    iobj:        {self.iobj}")
        print(f"    is_bound:    {self.is_bound}")
        print(f"    is_valid:    {self.is_valid()}")
        print(f"    p_read:      {self.p_read}")
        print(f"    p_update:    {self.p_update}")
        print(f"    p_delete:    {self.p_delete}")
        print( "    fields:")
        #pprint(self.schema)
        for k,v in self.fields.items():
            v.print()
            #pprint(v.fieldschema)
        print()


    def to_yaml(self):

        # NEXT : Rewrite ; aggregate YAML from each field (for various format)

        data = self.get_dict_for_export()
        data.pop('id', None)
        datastr =  yaml.dump([data], 
            allow_unicode=True, 
            Dumper=MyYamlDumper, 
            #default_style='|', 
            sort_keys=False)


        return datastr


    def cache_purge(self):
        cache.instance_purge(self.classname, self.keyname)

    @property
    def id(self):
        if self.iobj:
            return self.iobj.id


    @property
    def is_bound(self):
        ''' if object from DB in self.iobj '''
        if self.iobj:
            return True
        return False


    def get_related(self):

        instances =  DataEAV.objects.filter(
            format = 'schema:'+self.classname,
            value = self.keyname
        )

        # enrich with classname's displayname  (dict : classname=>class displayname)
        table = Schema.displayname_dict()
        reply = []
        for i in instances:
            displayname = table.get( i.classname, i.classname)
            i.classname_displayname = displayname
            reply.append(i)
        return reply


    def ordered_fields(self):

        def sort_key(item):
            return self.fields[item].order
        
        return sorted(self.fields, key = sort_key)
        
        
    def is_field_true(self, fieldname=None):
        if not fieldname:
            return False 
        if fieldname not in self.fields:
            return False 
        return self.fields[fieldname].is_field_true()


    def get_attribute_first(self, fieldname):

        r1 = self.get_attribute(fieldname)
        try:
            return r1[0]
        except:
            return ""


    def get_attribute(self, fieldname):
        '''  
        Returns a List []  of attribute value(s) from instance. 
        Convert to obj if FieldSchema  or SireneGroup or User ...
        '''

        if fieldname == "displayname":
            return [self.displayname]

        if fieldname == "keyname":
            return [self.keyname]

        if fieldname == "is_enabled":
            return [self.is_enabled]

        if fieldname not in self.fields:
            return

        return self.fields[fieldname].get_attribute()


    def set_field_value_single(self, fieldname=None, value=None):

        if not fieldname:
            return False
        if fieldname not in self.fields:
            return False
        self.fields[fieldname].value = [value]
        return True
        

    def has_read_permission(self, aaa=None):
        # iobj is a DB object

        try:
            if type(aaa["perms"]) is not list:
                return False
        except:
            return False

        if not self.is_bound:
            return False

        if "p_data_admin" in aaa["perms"]:
            return True

        # p_admin on classobj ? (allow direct & stop inheritance)
        if self.iobj.classobj.p_admin:
            if self.iobj.classobj.p_admin in aaa["perms"]:
                return True

        # instance permission
        if self.p_read:
            if self.p_read in aaa["perms"]:
                return True
            else:
                return False

        # default
        return has_read_permission_on_class(aaa=aaa, classobj=self.iobj.classobj)


    def has_edit_permission(self, aaa=None):
        # iobj is a DB object

        try:
            if type(aaa["perms"]) is not list:
                return False
        except:
            return False

        if not self.is_bound:
            return False

        if "p_data_admin" in aaa["perms"]:
            return True
    
        # p_admin on classobj: allow all on this class; no override 
        if self.iobj.classobj.p_admin:
            if self.iobj.classobj.p_admin in aaa["perms"]:
                return True
                
        # if defined, instance permission rules 
        if self.iobj.p_update:
            if self.iobj.p_update in aaa["perms"]:
                return True
            else:
                return False

        # default to class permission
        return has_edit_permission_on_class(aaa=aaa, classobj=self.iobj.classobj)



    def has_delete_permission(self, aaa=None):

        try:
            if type(aaa["perms"]) is not list:
                return False
        except:
            return False

        if not self.is_bound:
            return False

        if "p_data_admin" in aaa["perms"]:
            return True
    
        # p_admin on classobj: allow all on this class; no override 
        if self.iobj.classobj.p_admin:
            if self.iobj.classobj.p_admin in aaa["perms"]:
                return True
                
        # if defined, instance permission rules
        if self.p_delete:
            if self.p_delete in aaa["perms"]:
                return True
            else:
                return False

        # default to class permission
        return has_delete_permission_on_class(aaa=aaa, classobj=self.iobj.classobj)



    def get_recursive_content(self, fieldname=None, fieldmember=None, fieldrecurse=None, done=[]):
        ''' get list of Instance() from field "fieldname" , including
        - self instance
        - fieldmember *.fieldname
        - recurse on fieldsubgroup : fieldname, member, sub-subgroups ...
        '''

        # Example
        # -------
        # SiteGroup = { 
        #   sirene_notify =[ SireneGroup1, 2, ...]
        #   members = [site1, site2] 
        #   subgroups = [sitegroup1, sitegroup2 ...]
        #}
        #
        # for sitegroup_obj in message.notify_sitegroup.all():
        #     instance = Instance(iobj=sitegroup_obj)
        #     xlist = instance.get_recursive_content(
        #         fieldname="notify_group", 
        #         fieldmember="members", 
        #         fieldrecurse="subgroups",
        #         done=[])
        #
        # done[] contains list of iobj like self            


        reply=[]

        # recursive: already done ?
        if self in done:
            return

        done.append(self)

        # get field values from self
        xlist1 = self.get_attribute(fieldname)
        for z in xlist1:
            if z:
                if z not in done:
                    reply.append(z)
        #reply +=  xlist1

        # get all members from self
        xlist2 = []
        if fieldmember:
            # if fieldmember in self.schema:
            if fieldmember in self.fields:
                xlist2 = self.fields[fieldmember].get_attribute()

        # get field values from members
        for item in xlist2:
            # try to get an instance struct
            # NOTA: won't work on SireneGroup objects ; use SireneGroups for that purpose
            if type(item) is DataInstance:
                instance = Instance(iobj=item)
                if instance:
                    xlist3 = instance.get_attribute(fieldname)
                    for z in xlist3:
                        if z:
                            if z not in done:
                                reply.append(z)

        # get all subgroup (fieldrecurse field content)
        xlist4 = []
        if fieldrecurse:
            if fieldrecurse in self.fields:
                xlist4 = self.fields[fieldrecurse].get_attribute()

        # recurse
        for item in xlist4:
            if type(item) is DataInstance:
                if item in done:
                    continue
                instance = Instance(iobj=item)
                if instance:
                    xlist5 = instance.get_recursive_content(
                        fieldname=fieldname,
                        fieldmember=fieldmember,
                        fieldrecurse=fieldrecurse,
                        done=done
                        )
                    for z in xlist5:
                        if z:
                            if z not in done:
                                reply.append(z)
                    #reply += xlist5


        reply = list(set(reply))
        return reply


    # set options
    # -----------
    def set_options(self):
        
        # empty instance
        if not self.schema2:
            return
        
        # keyname_mode
        if self.schema2.keyname_mode == 'auto':
            if not self.keyname:
                self.keyname = Schema.create_keyname()

                


    # expand / compute injected fields
    # --------------------------------

    def expand_injected(self):

        # Refresh all injected / external / enumerate / schema 

        # if new empty Instance, skip update
        if not self.keyname:
            return

        # 1 - remove injected fields (from enumerate / schema)
        # self.fields will be modified so separate loop inventory and processing
        remove = []
        for fieldname,field in self.fields.items():
            if field.is_injected:
                remove.append(fieldname)
        for v in remove:
            self.fields.pop(v)

  
        # 2 - resolve external to source
        # self.fields will be modified so separate loop inventory and processing
        external = []
        for fieldname,field in self.fields.items():
            if field.dataformat == "external":    
                external.append(fieldname)

        # processing
        for fieldname in external:
            #field = self.fields[fieldname]
            # recurse here ; will inject resolved field as schema fields
            self.inject_resolved_external_field(fieldname)

        # 3 - schema : inject subfields
        self.inject_schema_subfields()

        # 4 - enumerate : inject subfields
        self.inject_enumerate_subfields()

        return



    # -------------------------------------------------
    # compute with recurse for a single field
    # inject computed field in self.fields
    # -------------------------------------------------
    def inject_resolved_external_field(self, fieldname):
    
        try:
            field = self.fields[fieldname]
        except:
            return

        # fieldname must be an external (to a schema)
        if field.dataformat != "external":
            return

        parent_fieldname = field.get_parent_fieldname()
        if not parent_fieldname:
            return

        parent_field = self.fields[parent_fieldname]
        if parent_field.dataformat != "schema":
            return

        remote_classname = parent_field.get_classname()
        if not remote_classname:
            return


        # NEXT : support multi-value
        try:
            remote_instance_name = parent_field.value[0]
        except:
            return

        # NEXT : add depth to avoid recurse loops
        # don't expand (perf) ; will recurse on non expanded field
        remote_instance = Instance.load_from_names(classname=remote_classname, keyname=remote_instance_name)
        if not remote_instance:
            return
        if not remote_instance.iobj:
            return

        # get remote fieldname
        remote_fieldname = field.get_remote_fieldname()
        if not remote_fieldname:
            return

        # 2 options here
        # either remote_fieldname is a schema , or "_"+remote_field is an external
        if remote_fieldname in remote_instance.fields:
            # schema
            remote_fieldname2 = remote_fieldname
            remote_field = remote_instance.fields[remote_fieldname]
        elif "_" + remote_fieldname in remote_instance.fields:
            # external
            remote_fieldname2 = "_" + remote_fieldname
            remote_field = remote_instance.fields[remote_fieldname2]
        else:
            remote_field = None

        if not remote_field:
            return

        remote_field_dataformat = remote_field.dataformat
  
        # remote field is an external => recurse (again)
        if remote_field_dataformat == "external":
            remote_instance.inject_resolved_external_field(remote_fieldname2)
            # access remote resolved field (without prefix _ )
            try:
                # remote_fieldname is without _
                newvalue = remote_instance.fields[remote_fieldname].get_value()
                newclassname = remote_instance.fields[remote_fieldname].get_classname()
            # bad structure
            except:
                return

        # remote field is a schema => DONE, it's the source
        elif remote_field_dataformat == "schema": 
            newvalue = remote_field.get_value()
            newclassname = remote_field.get_classname()
        # bad structure
        else:
            return

        # inject external schema field w/ name = external fieldname without prefix _
        new_fieldname = field.fieldname[1:]
        new_json = {new_fieldname:newvalue}
        # get subfields
        subfields = field.get_subfields()
        new_dataformat_ext = newclassname + ' ' + ' '.join(subfields)
        new_schema = {
            'displayname':field.displayname,
            'description':new_fieldname, 
            'is_multi': False,  
            'dataformat': 'schema',
            'dataformat_ext': new_dataformat_ext,
            'cardinal_min':0,
            'cardinal_max':1,
            'is_injected': True,
            'injected_type':'external',
            'page': field.page,
            'order': field.order,
            }
        self.fields[new_fieldname] = FieldSchema(new_fieldname, new_schema, new_json)
        return



    def inject_schema_subfields(self):

        # collect all schema fields to process (2 stage : fields will be altered by injection)
        schema_fieldnames = []

        for fieldname, field in self.fields.items():
            if field.dataformat == "schema":
                schema_fieldnames.append(fieldname)
       

        # loop over identified schema fields
        for fieldname in schema_fieldnames:
            field = self.fields[fieldname]

            subfieldnames = field.get_subfields()
            if len(subfieldnames) == 0:
                continue

            target_classname = field.get_classname()
            target_instancename = field.get_first_value()
            # NEXT : multi : field__xx.0, field_xx.1 ...
            target_instance = Instance.load_from_names(classname=target_classname, keyname=target_instancename)
            if not target_instance:
                continue

            for fn in subfieldnames:
                
                new_fieldname = fieldname + "__" + fn
                new_displayname = field.displayname + f"({fn})"
                # try a regular target field:
                if fn in target_instance.fields:
                    
                    target_field = target_instance.fields[fn]

                    # exclude some dataformat we can't inject
                    if target_field.dataformat in ['schema', 'external']:
                        continue

                    new_dataformat = target_field.dataformat
                    new_schema = {
                        'displayname': new_displayname, 
                        'description': new_displayname, 
                        'dataformat': new_dataformat,
                        'dataformat_ext': target_field.dataformat_ext,
                        'cardinal_min': target_field.cardinal_min,
                        'cardinal_max': target_field.cardinal_max,
                        'is_injected': True,
                        'injected_type':'schema',
                        'page': field.page, 
                        'order': field.order, 
                        }                    

                    try:
                        new_list = target_field.get_value()
                        new_json = {new_fieldname:new_list}
                    except:
                        continue
                            
                    constructor = FIELD_CONSTRUCTOR_TABLE.get(new_dataformat, FieldString)
                    self.fields[new_fieldname] = constructor(new_fieldname, new_schema, new_json)

                # try a built-in attribute (display_name ...)
                else:
                    pass
                # TODO : inject displayname, keyname




    def inject_enumerate_subfields(self):

        enumerate = []
        for fieldname, field in self.fields.items():
            if field.dataformat == "enumerate":
                enumerate.append(fieldname)

        for fieldname in enumerate:
            field = self.fields[fieldname]
            subfields = field.get_subfields()

            for k,value in subfields.items():

                new_fieldname = fieldname + "__" + k
                try:
                    (t,v) = value
                except:
                    continue
                new_json = {new_fieldname:[v]}
                new_schema = {
                    'displayname':new_fieldname, 
                    'description':new_fieldname, 
                    'dataformat': 'string',
                    'dataformat_ext':'safe',
                    'cardinal_min':0,
                    'cardinal_max':1,
                    'is_injected': True,
                    'injected_type':'enumerate',
                    'page': field.page,
                    'order': field.order,
                    }

                if t == "int":
                    new_schema["dataformat"] = "int"
                    self.fields[new_fieldname] = FieldInt(new_fieldname, new_schema, new_json)

                elif t =="booelan":
                    new_schema["dataformat"] = "boolean"
                    self.fields[new_fieldname] = FieldBoolean(new_fieldname, new_schema, new_json)

                elif t == "float":
                    new_schema["dataformat"] = "float"
                    self.fields[new_fieldname] = FieldFloat(new_fieldname, new_schema, new_json)

                else:
                    self.fields[new_fieldname] = FieldString(new_fieldname, new_schema, new_json)

                # NEXT : date, ipv4, datefime, time



    # --------


    def get_dict_for_ui_detail(self, skip_external=True, skip_injected=True, skip_enumerate=True):
        ''' dict suited for list template  ; NESTED PAGE and FLAT '''

        # ui = { 
        #   keyname:"xxxx"
        #   displayname:"xxx"
        #   is_enabled:True/False"
        #   p_*3

        # First Level / Flat mode
        # ------------------------
        #   fieldname: { DATAPOINT }
        #   fieldname: { DATAPOINT }
        #   ...

        # Nested / Page
        # -------------
        # instance_ui["PAGES"][page][order]= []
        #
        #   "PAGES": {  
        #     "(page)1": { "(order)100": [ {DATAPOINT}, {}, ... ], 
        #                  "(order)101": [ {}, {}, ... ], 
        #     "(page)2": { }, 
        #    ... 
        # }
        
        # DATAPOINT = {
            # datapoint["fieldname"] = self.fieldname
            # datapoint["displayname"] = self.displayname
            # datapoint["description"] = self.description
            # datapoint["dataformat"] = self.dataformat
            # datapoint["dataformat_ext"] = self.dataformat_ext
            # datapoint["is_multi"] = self.is_multi()
            # datapoint["bigset"] = False
            # datapoint["schema"] = CLASSNAME   (for schema field)
            # datapoint["value"] = ''
        # }
        # VALUE : depend on field dataformat (ex. ", ".join(values)")

        instance_ui = {}

        # SPECIAL fields P7
        instance_ui["id"] = self.id
        instance_ui["keyname"] = self.keyname
        instance_ui["displayname"] = self.displayname
        instance_ui["p_read"] = self.p_read
        instance_ui["p_update"] = self.p_update
        instance_ui["p_delete"] = self.p_delete
        instance_ui["is_enabled"] = self.is_enabled

        # loop over instance.fields
        instance_ui["PAGES"] = {}

        for fieldname in self.ordered_fields():
            field = self.fields[fieldname]


            if skip_external:
                if field.dataformat == "external":
                    continue
            if skip_injected:
                if field.is_injected:
                    continue
            if skip_enumerate:
                if field.injected_type == "enumerate":
                    continue

            order = field.order
            page = field.page
            if not page:
                page = self.keyname

            if page not in instance_ui["PAGES"]:
                instance_ui["PAGES"][page] = {}
            if order not in instance_ui["PAGES"][page]:
                instance_ui["PAGES"][page][order]= []

            datapoint = field.get_datapoint_ui_detail()

            # nested structure PAGE/ORDER
            instance_ui["PAGES"][page][order].append(datapoint)

            # append also at first level
            instance_ui[fieldname] = copy.deepcopy(datapoint)

        return instance_ui


    
    def get_dict_for_ui_form(self):
        # used for edit in form : new or edit

        instance_ui = {}

        # SPECIAL fields
        instance_ui["id"] = self.id
        instance_ui["keyname"] = self.keyname
        instance_ui["displayname"] = self.displayname
        instance_ui["p_read"] = self.p_read
        instance_ui["p_update"] = self.p_update
        instance_ui["p_delete"] = self.p_delete
        instance_ui["is_enabled"] = self.is_enabled


        # loop over instance.fields
        instance_ui["PAGES"] = {}
        for fieldname in self.ordered_fields():
            field = self.fields[fieldname]

            # don't edit injected fields
            if field.is_injected:
                continue

            # don't edit external fields
            if field.dataformat == "external":
                continue

            order = field.order
            page = field.page
            if not page:
                page = self.keyname

            if page not in instance_ui["PAGES"]:
                instance_ui["PAGES"][page] = {}
            if order not in instance_ui["PAGES"][page]:
                instance_ui["PAGES"][page][order]= []

            datapoint = self.fields[fieldname].get_datapoint_ui_edit()
            instance_ui["PAGES"][page][order].append(datapoint)

        return instance_ui


    
    def get_dict_for_export(self):
        
        # {
        #   classname: XX
        #   keyname: XX
        #   displayname:"xxx"
        #   is_enabled:True/False"
        #   "attribname": DATAPOINT, 
        #   "attribname": ... 
        # }

        # DATAPOINT = value / "value"
        # DATAPOINT = ["v1", "v2", ...]

        instance_export = {}

        # SPECIAL fields
        instance_export["id"] = self.id
        instance_export["classname"] = self.classname
        instance_export["keyname"] = self.keyname
        instance_export["displayname"] = self.displayname
        if self.p_read:
            instance_export["p_read"] = self.p_read
        if self.p_update:
            instance_export["p_update"] = self.p_update
        if self.p_delete:
            instance_export["p_delete"] = self.p_delete
        instance_export["is_enabled"] = self.is_enabled

        for fieldname in self.ordered_fields():
            field = self.fields[fieldname]
            datapoint = field.get_datapoint_for_export()
            cardinal, cardinal_min, cardinal_max = field.get_cardinal3()
            if not (cardinal_min == 0 and cardinal == 0):
                instance_export[fieldname] = datapoint

        return instance_export


    def get_csv_columns(self):

        csv_columns = ["classname","keyname","displayname","is_enabled"]
        # ,"p_read","p_update","p_delete
        for fieldname in self.ordered_fields():
            field = self.fields[fieldname]
            if field.is_multi():
                continue
            if field.is_injected:
                continue
            if fieldname.startswith('_'):
                continue
            if field.dataformat == "text":
                continue
            csv_columns.append(fieldname)
        return csv_columns
                                    

    def get_csv_line(self, csv_columns):
        
        # /!\ must match get_csv_columns
        line = [self.classname, self.keyname, self.displayname,self.is_enabled ]
        # self.p_read, self.p_update,self.p_delete
        for fieldname in self.ordered_fields():
            if fieldname not in csv_columns:
                continue
            line.append(self.fields[fieldname].get_csv_cell())
        
        return line


    # new 3.20 : merged new & edit method
    def merge_request(self, request, aaa=None):

        # keyname : editable only for new objects (PK in DB)
        if not self.is_bound:
            if self.schema2.keyname_mode == 'edit':
                self.keyname = request.POST.get("keyname", "")

        # displayname (check in is_valid + template auto escaping)
        self.displayname = request.POST.get("displayname", "")

        # is_enabled
        if "is_enabled" in request.POST:
            self.is_enabled = request.POST["is_enabled"] in TRUE_LIST
        else:
            self.is_enabled = False

        # permissions
        if type(aaa) is dict:
            if 'perms' in aaa:
                if "p_data_security_edit" in aaa['perms']:
                    if "p_read" in request.POST:
                        self.p_read = request.POST["p_read"]
                    if "p_update" in request.POST:
                        self.p_update = request.POST["p_update"]
                    if "p_delete" in request.POST:
                        self.p_delete = request.POST["p_delete"]


        for fieldname,field in self.fields.items():

            # don't merge injected fields
            if field.is_injected:
                continue

            # don't merge external fields
            if field.dataformat == "external":
                continue     

            self.fields[fieldname].merge_request(request)

        # injected fields
        self.expand_injected()

        # options
        self.set_options()





    def merge_import(self, data, aaa=None):

        # keyname if new instance only, and not auto computed keyname
        if not self.is_bound:
            if self.schema2.keyname_mode == 'edit':
                # keyname may be absent 
                # (new Instance with initial keyname, merge additional data w/o keyname)
                if 'keyname' in data:
                    self.keyname = data.get("keyname", None)

        # displayname
        if 'displayname' in data:
            self.displayname = data["displayname"]

        # is_enabled
        if "is_enabled" in data:
            self.is_enabled = data["is_enabled"] in TRUE_LIST
        #else:
        #    self.is_enabled = False

        # permissions
        if type(aaa) is dict:
            if 'perms' in aaa:
                if "p_data_security_edit" in aaa['perms']:
                    if "p_read" in data:
                        self.p_read = data["p_read"]
                    if "p_update" in data:
                        self.p_update = data["p_update"]
                    if "p_delete" in data:
                        self.p_delete = data["p_delte"]

        for fieldname,field in self.fields.items():

            # don't merge injected fields
            if field.is_injected:
                continue

            # don't merge external fields
            if field.dataformat == "external":
                continue     

            # special case : Boolean are not in POST request if False
            if fieldname in data:
                fielddata = data[fieldname]
                self.fields[fieldname].merge_import(fielddata)

        
        # injected fields
        self.expand_injected()

        # options
        self.set_options()




    def is_valid(self):

        reply = True

        # keyname
        if not self.keyname:
            self.errors.append(_("missing keyname"))
            return False
        if len(self.keyname) == 0:
            self.errors.append(_("keyname empty"))
            return False

        # classobj
        if not self.classobj:
            self.errors.append(_("missing classobj"))
            return False
        if not self.classname:
            self.errors.append(_("missing classname"))
            return False
        if len(self.classname)==0:
            self.errors.append(_("empty classname"))
            return False
        if self.classobj.keyname != self.classname:
            self.errors.append(_("classname/classobj mismatch"))
            return False
        
        # NEXT : classname exist in DB ?
        # NEXT:  displayname = safe string for display ?

        # is_enabled
        if type(self.is_enabled) is not bool:
            return False

        # NEXT - check permissions ?

        for fieldname,field in self.fields.items():
            r = field.is_valid()
            if not r:
                reply = False
                #err = fieldname
                err = field.displayname
                self.errors.append(str(err))
                # NEXT:  use field.errors info

        return reply

    # ACTIONs : update Instance() and iobj ; call save to DB
    # -------------------------------------------------------

    def enable(self):
        if not self.is_bound:
            return False
        self.cache_purge()
        self.iobj.is_enabled=True
        self.is_enabled=True
        return self.save()


    def disable(self):
        if not self.is_bound:
            return False
        self.cache_purge()
        self.is_enabled=False
        self.iobj.is_enabled=False
        return self.save()


    def init(self):
        '''  init only if not existing  '''

        # already exists in struct?
        if self.iobj:
            self.errors.append(_("init - already created - can't recreate"))
            return False

        # instance already in DB ?
        iobj = get_instance_by_name(iname=self.keyname, classname=self.classname)
        if iobj:
            self.errors.append(_("init - instance already in DB - can't recreate"))
            return False

        # create new instance
        self.iobj = DataInstance()
        self.iobj.classobj = self.classobj
        # V3.19
        self.iobj.classname = self.classobj.keyname

        self.update()



    def create(self):
        ''' Create or update new Instance in DB from 'instance' '''

        # compute/update keyname, ...
        self.set_options()

        # create if doesn't exist
        if not self.is_bound:
            iobj = DataInstance()
            iobj.classobj = self.classobj               # deprecated
            iobj.classname = self.classobj.keyname      # new V3.19
            iobj.keyname = self.keyname
            iobj.is_enabled = self.is_enabled
            iobj.p_read = self.p_read
            iobj.p_update = self.p_update
            iobj.p_delete = self.p_delete
            #iobj.save()
            self.iobj = iobj

        return self.update()
    

    def update(self):
        ''' update self.iobj with Instance struct , and save to DB'''

        if not self.is_bound:
            return False
        
        self.cache_purge()

        self.set_options()

        self.iobj.keyname = self.keyname
        self.iobj.displayname = self.displayname
        self.iobj.is_enabled = self.is_enabled
        self.iobj.p_read = self.p_read
        self.iobj.p_update = self.p_update
        self.iobj.p_delete = self.p_delete
        # V3.19 - populate new classname field
        self.iobj.classname = self.classobj.keyname

        data = {}
        for fieldname,field in self.fields.items():
            
            # don't store injected fields
            if field.is_injected:
                continue

            field_datalist = self.fields[fieldname].get_value()
            # don't store empty fields in DB
            if len(field_datalist) == 0:
                continue

            # external field : restore fieldname without  _ prefix
            if field.dataformat == "external":
                original_fieldname = fieldname[1:]
                data[original_fieldname] = field_datalist
            else:
                data[fieldname] = field_datalist

        # NEXT : strict mode : self.is_valid() must be True

        try:
            self.iobj.data_json = json.dumps(data, indent=2, ensure_ascii=False)
        except:
            print("Invalid JSON in update()")
            return False

        return self.save()


    # real write to DB
    # -----------------
    # NEXT - bulk write to DataInstanceSearch for searchable fields

    def delete(self):
        # place #1 for real write to DB
        if not self.is_bound:
            return False
        self.cache_purge()
        try:
            self.iobj.delete()
        except Exception as e:
            print(f"DB delete failed for {self.classname}:{self.keyname} - {e}")
            return False
        
        self.iobj = None
        self.last_update = timezone.now()

        self.update_eav()
        return True
    

    def save(self):
        # place #2 for real write to DB

        self.cache_purge()

        if not self.iobj:
            print("!! no iobj in save()")
            return False

        # V3.19 - populate new classname field
        self.iobj.classname = self.classobj.keyname


        self.last_update = timezone.now()
        self.iobj.last_update = self.last_update

        try:
            self.iobj.save()
        except Exception as e:
            print(f"save() to DB failed: {e}")
            return False
        
        self.update_eav()
        return True


    def update_eav(self):

        # delete older EAV entries
        DataEAV.objects.filter(
            classname = self.classname,
            keyname = self.keyname
        ).delete()

        # self.iobj must be deleted
        if not self.iobj:
            return

        # save at least one entry with keyname
        eavobj = DataEAV(
            iid = self.id,
            classname = self.classname,
            keyname = self.keyname,
            displayname = self.displayname,
            fieldname = 'keyname',
            format = 'string',
            value = self.keyname,
            last_update = self.last_update
        )
        try:
            eavobj.save()
        except:
            pass
        

        # Loop over fields (if any)
        for fieldname,field in self.fields.items():

            # don't store injected fields
            if field.is_injected:
                continue

            # external field also
            if field.dataformat == "external":
                continue

            values = field.get_eav_list()
            format = field.get_eav_format()

    

            for v in values:

                if len(v) > 1000:
                    # v ="<<< removed - too big >>>"
                    continue

                if len(v) == 0:
                    continue

                eavobj = DataEAV(
                    iid = self.id,
                    classname = self.classname,
                    keyname = self.keyname,
                    displayname = self.displayname,
                    fieldname = fieldname,
                    format = format,
                    value = v,
                    last_update = self.last_update
                )

                try:
                    eavobj.save()
                except:
                    pass



# --------------------------------------------------------
# LOADER / IMPORT
# Global LOADER : class, schema, instance, static
# --------------------------------------------------------

def load_schema(datadict=None, verbose=True, aaa=None):

    # DEPRECATED: options, icon,  p_* 
    META = ["keyname", "displayname", "is_enabled", "_options", "_sections",
            "icon", "order", "page", "options",
            "p_read", "p_create", "p_update", "p_delete", "p_admin",
            ]


    if not datadict:
        return

    if not aaa:
        log(WARNING, aaa=aaa, app="data", view="schema", action="load", status="DENY", data=_("Not allowed")) 
        return

    keyname = datadict.get("keyname", None)
    if not keyname: 
        return

    classobj = load_class_part(keyname, datadict, verbose=verbose, aaa=aaa)
    if not classobj:
        return

    classname = classobj.keyname

    # fields
    for fieldname, fielddata in datadict.items():
        if fieldname in META:
            continue      
        load_field_definition(classname, fieldname, fielddata, verbose=verbose, aaa=aaa)

    return classobj


# loads class definition (fields not loaded here)
def load_class_part(classname, classdata, verbose=True, aaa=None):

    if not type(classname) is str:
        return

    if not type(classdata) is dict:
        return

    if not len(classname)>0: 
        return

    if not aaa:
        log(WARNING, aaa=aaa, app="data", view="class", action="load", status="DENY", data=_("Not allowed")) 
        return   
   
    # action = force_action
    # if not action:
    action = classdata.get("_action", "create")

    # No cache, direct access for Load
    classobj = DataClass.objects.filter(keyname=classname).first()

    if action == "delete":
        if not has_schema_delete_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        if classobj:
            classobj.delete()
            return classobj
        else:
            return

    elif action == "disable":
        if not has_schema_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        if classobj:
            classobj.is_enabled = False
            classobj.save()
            return classobj

    elif action == "enable":
        if not has_schema_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        if classobj:
            classobj.is_enabled = True
            classobj.save()
            return classobj

    elif action == "update":
        if not has_schema_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        # no creation, see below

    elif action == "init": 
        if not has_schema_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        # skip if alreay created
        if classobj:
            # return None to skip creating fields
            return classobj
        classobj=DataClass(keyname=classname)
        classobj.save()

    elif action == "create":
        if not has_schema_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="schema", action=action, status="DENY", data=_("Not allowed")) 
            return classobj
        if not classobj:    
            classobj=DataClass(keyname=classname)
            classobj.save()
    else:
        # unknown action
        log(WARNING, aaa=aaa, app="data", view="schema", action="unknown", status="DENY", data=_("Unknown action")) 
        if verbose:
            print(f"  !!! unknown action {action} for class {classname}")
        return classobj

    # update for all remaining actions
    # --------------------------------
    if not has_schema_update_permission(aaa=aaa):
        log(WARNING, aaa=aaa, app="data", view="schema", action="update", status="DENY", data=_("Unknown action")) 
        return classobj

    if not classobj:
        return

    if "displayname" in classdata:
        classobj.displayname = classdata.get("displayname","")

    if "order" in classdata:
        classobj.order = classdata.get("order",100)

    if "page" in classdata:
        classobj.page = classdata.get("page")

    if "is_enabled" in classdata:
        classobj.is_enabled = classdata.get("is_enabled") in TRUE_LIST


    # update permissions ; 3.19- => deprecated
    if "p_read" in classdata:
        classobj.p_read = classdata.get("p_read")
    if "p_create" in classdata:
        classobj.p_create = classdata.get("p_create")
    if "p_update" in classdata:
        classobj.p_update = classdata.get("p_update")
    if "p_delete" in classdata:
        classobj.p_delete = classdata.get("p_delete")
    if "p_admin" in classdata:
        classobj.p_admin = classdata.get("p_admin")

    # new 3.20 _options builtin YAML structure
    # ----------------------------------------
    options = classdata.get("_options", {})
    # copy YAML in DB Text field
    classobj.options = yaml.dump(options, allow_unicode=True, sort_keys=True)


    # permissions V3.20+
    if "p_read" in options:
        classobj.p_read = options.get("p_read")
    if "p_create" in options:
        classobj.p_create = options.get("p_create")
    if "p_update" in options:
        classobj.p_update = options.get("p_update")
    if "p_delete" in options:
        classobj.p_delete = options.get("p_delete")
    if "p_admin" in options:
        classobj.p_admin = options.get("p_admin")


    classobj.save()

    log(INFO, aaa=aaa, app="data", view="schema", action=action, status="OK", data=f"{classname}")

    return classobj



# load Schema fields in DB
def load_field_definition(classname, fieldname, fielddata, verbose=True, aaa=None):

    # refuse to load for a non-existent Class 
    #classobj=DataClass(keyname=classname)
    #classobj=get_class_obj(classname=classname)
    classobj = get_class_by_name(classname)
    if not classobj:
        return

    if not type(fieldname) is str:
        return

    if not type(fielddata) is dict:
        return

    if not len(fieldname)>0: 
        return

    if not aaa:
        log(WARNING, aaa=aaa, app="data", view="field_def", action="load", status="DENY", data=_("Not allowed")) 
        return
    
    action = fielddata.get("_action", "create")

    fieldobj = DataSchema.objects.filter(classname=classname, keyname=fieldname).first()

    if action == "delete":
        if not has_schemafield_delete_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="delete", status="DENY", data=_("Not allowed")) 
            return        
        if fieldobj:
            fieldobj.delete()
            if verbose:
                print(f"  field delete - {classname}:{fieldname}")            
            return fieldobj

    elif action == "disable":
        if not has_schemafield_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="disable", status="DENY", data=_("Not allowed")) 
            return        
        if fieldobj:
            fieldobj.is_enabled = False
            fieldobj.save()
            if verbose:
                print(f"  field disable - {classname}:{fieldname}")            
            return fieldobj

    elif action == "enable":
        if not has_schemafield_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="enable", status="DENY", data=_("Not allowed")) 
            return        
        if fieldobj:
            fieldobj.is_enabled = True
            fieldobj.save()
            if verbose:
                print(f"  field enable - {classname}:{fieldname}")            
            return fieldobj

    elif action == "update":
        if not has_schemafield_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="update", status="DENY", data=_("Not allowed")) 
            return        
        # no creation, see below for update
        if not fieldobj:
            return

    elif action == "init": 
        if not has_schemafield_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="init", status="DENY", data=_("Not allowed")) 
            return        
        # skip if alreay created
        if fieldobj:
            return fieldobj
        fieldobj = DataSchema()
        fieldobj.classobj = classobj
        fieldobj.classname = classname
        fieldobj.keyname = fieldname
        fieldobj.save()
        if verbose:
            print(f"  field init - {classname}:{fieldname}")            


    elif action == "create":
        if not has_schemafield_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="data", view="field_def", action="create", status="DENY", data=_("Not allowed")) 
            return        
        if not fieldobj:
            fieldobj = DataSchema()
            fieldobj.classobj = classobj
            fieldobj.classname = classname
            fieldobj.keyname = fieldname
            fieldobj.default = fielddata.get("default", "")
            fieldobj.save()
            if verbose:
                print(f"  field create - {classname}:{fieldname}")            
    else: 
        # unknown action
        log(WARNING, aaa=aaa, app="data", view="field_def", action="unknown", status="DENY", data=_("Unknown action")) 
        if verbose:
            print(f"  ERR unknown field action: {action} for {classname}:{fieldname}")
        return fieldobj


    # update provided params if provided only !
    if not fieldobj:
        return

    if not has_schemafield_update_permission(aaa=aaa):
        log(WARNING, aaa=aaa, app="data", view="field_def", action="update", status="DENY", data=_("Not allowed")) 
        return

    if "displayname" in fielddata:
        fieldobj.displayname = fielddata.get("displayname")

    if "description" in fielddata:
        fieldobj.description = fielddata.get("description")

    if "is_enabled" in fielddata:
        fieldobj.is_enabled = fielddata.get("is_enabled") in TRUE_LIST

    if "order" in fielddata:
        fieldobj.order = int( fielddata.get("order") )
    if "page" in fielddata:
        fieldobj.page = fielddata.get("page")

    if "dataformat" in fielddata:
        fieldobj.dataformat = fielddata.get("dataformat")
    if "dataformat_ext" in fielddata:
        fieldobj.dataformat_ext = fielddata.get("dataformat_ext")
    if "default" in fielddata:
        fieldobj.default_value = fielddata.get("default")
    if "cardinal_min" in fielddata:
        fieldobj.cardinal_min = int( fielddata.get("cardinal_min") )
    if "cardinal_max" in fielddata:
        fieldobj.cardinal_max = int( fielddata.get("cardinal_max") )

    fieldobj.save()
    log(INFO, aaa=aaa, app="data", view="field_def", action=action, status="OK", data=f"{classname} - {fieldname}")

    if verbose:
        print(f"  field update - {classname}:{fieldname}")
    return fieldobj



def load_instance(datadict=None, verbose=True, aaa=None):

    if not datadict:
        return

    if not aaa:
        log(WARNING, aaa=aaa, app="data", view="instance", action="load", status="DENY", data=_("Not allowed")) 
        return
    
    classname = datadict.get("classname", None)
    if not classname: 
        print(f"ERR - missing classname: {datadict}")
        return

    classobj = get_class_by_name(classname)
    if not classobj:
        print(f"ERR - unknown classname: {classname}")
        return


    action = datadict.get("_action", "create")

    keyname = datadict.get("keyname", None)

    # instance = Instance.load_from_names(classname=classname, keyname=keyname)
    # if not instance:
    #     log(WARNING, aaa=aaa, app="data", view="instance", action="load", status="KO", data=f"couldn't create {classname}")
    #     return
   
    # # auto or from_field... : keyname not mandatory in datadict
    # if instance.schema2.keyname_mode == 'edit':
    #     if not keyname: 
    #         print(f"ERR - missing keyname for {classname}: {datadict}")
    #         return


    if action == "delete":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if instance:
            if instance.has_delete_permission(aaa=aaa):
                instance.delete()
                if verbose:
                    print(f"instance ({action}) - {classname}:{keyname}")
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not found"))
        return


    elif action == "disable":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if instance:
            if instance.has_edit_permission(aaa=aaa):
                instance.disable()
                if verbose:
                    print(f"instance ({action}) - {classname}:{keyname}")
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not found"))
        return


    elif action == "enable":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if instance:
            if instance.has_edit_permission(aaa=aaa):
                instance.enable()
                if verbose:
                    print(f"instance ({action}) - {classname}:{keyname}")
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not found"))
        return
            

    # init only == new instance if doesn't already exist
    elif action == "init":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if instance:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Already exists"))
            return
        instance = Instance.from_classname(classname=classname)
        if instance:
            if has_create_permission_on_class(aaa, classobj=classobj):
                instance.merge_import(datadict)
                instance.init()
                if verbose:
                    print(f"instance ({action}) - {classname}:{keyname}")
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Couldn't create"))
        return
    

    # create and/or update if exists
    elif action == "create":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if not instance:
            instance = Instance.from_classname(classname=classname)
        if instance:
            if has_create_permission_on_class(aaa, classobj=classobj):
                instance.merge_import(datadict)
                instance.create()
                if verbose:
                    print(f"instance ({action}) - c={classname} k={instance.keyname} h={instance.id}")                
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Couldn't create"))
        return


    # don't create, update only if exists
    elif action == "update":
        instance = Instance.load_from_names(classname=classname, keyname=keyname, expand=False)
        if instance:
            if instance.has_edit_permission(aaa=aaa):
                instance.merge_import(datadict)
                instance.update()
                if verbose:
                    print(f"instance ({action}) - {classname}:{keyname}")                
                return instance
            else:
                log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not allowed"))
        else:
            log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Not found"))
        return

    # ubknown action
    else:
        log(WARNING, aaa=aaa, app="data", view="instance", action=action, status="DENY", data=_("Unknown action"))
        return


  






