# (c) cavaliba.com - data - views.py

import re
import yaml 
import json 

from django.shortcuts import render, redirect
from django.http import HttpResponse
from django.contrib import messages
from django.utils.translation import gettext as _
from django.db.models import Q


from app_home.log import log, DEBUG, INFO, WARNING, ERROR, CRITICAL
from app_user.aaa import start_view
from app_home.configuration import get_configuration
import app_home.cache as cache

#from .models import DataInstance

from .data import Instance
from .data import get_classes
from .data import get_class_by_name
from .data import get_schema

# export >> exporter.py
# from .exporter import data_yaml_response
# from .exporter import data_json_response

from .forms import DataUploadForm
from .loader import load_broker
from .pipeline import list_pipelines
from .pipeline import apply_pipeline
from .dataview import dataview_for_request
from .paginator import paginator_for_request
from .search import get_query_from_request
from .search import get_instance_from_query

from .permissions import has_read_permission_on_class
from .permissions import has_read_permission_on_instance
from .permissions import has_delete_permission_on_class
from .permissions import has_delete_permission_on_instance
from .permissions import has_edit_permission_on_instance
from .permissions import has_create_permission_on_class
from .permissions import has_export_permission_on_class
from .permissions import has_import_permission_on_class



# ----------------------------------------------
# list of Classes
# ----------------------------------------------
def private(request):

    context = start_view(request, app="data", view="private", noauth="app_sirene:index", 
        perm="p_data_read", noauthz="app_home:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    dataclasses = get_classes(is_enabled=True)
    filtered = []

    # Permission
    for classobj in dataclasses:    
        if has_read_permission_on_class(aaa=aaa, classobj=classobj):
            filtered.append(classobj)


    # page/order for UI
    # -----------------
    # [  [page1, [class1, class2,  ...] , [ page2, [...] ] ,  ... ]
    paginated = []
    pagelist = []
    index = {}   # page => [class1, class2]
    default_name = get_configuration("home", "GLOBAL_APPNAME")

    for classobj in filtered:
        # order = element.order
        page = classobj.page
        if not page:
            page = default_name
        if len(page) == 0:
            page = default_name
        if page not in index:
            index[page] = []
            pagelist.append(page)
        index[page].append(classobj)    

    for p in pagelist:
        paginated.append( [p, index[p] ])


    log(DEBUG, aaa=aaa, app="data", view="private", action="list", status="OK", data="{} classes".format(len(filtered)))

    context["dataclasses"] = filtered
    context["paginated"] = paginated
    return render(request, 'app_data/private.html', context)

# -------------------------------------------------------------------------
# Instance List 
# -------------------------------------------------------------------------
# V3.10 : always bigset

def instance_list(request, classname=None):

    context = start_view(request, app="data", view="instance_list", noauth="app_sirene:index", 
        perm="p_data_read", noauthz="app_data:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    classobj = get_class_by_name(classname)
    if not classobj:
        redirect("app_data:private")

    # permission
    if not has_read_permission_on_class(aaa=aaa, classobj=classobj):
        log(WARNING, aaa=aaa, app="data", view="instance", action="list", status="KO", data=_("Not allowed")) 
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        return redirect("app_data:private")

    paginator = paginator_for_request(request=request, classobj=classobj, update_session=True)
    if not paginator:
        return redirect("app_data:private")

    query = get_query_from_request(request)

    # instances
    instances = get_instance_from_query(classobj=classobj, query=query, offset=paginator.offset, limit=paginator.limit)

    # apply permissions - NEXT: number ot items can be reduced by permission filtering ; add more ?
    filtered = []
    for i in instances:
        if has_read_permission_on_instance(aaa=aaa, iobj=i):
            filtered.append(i)

    # apply dataview
    dataview = dataview_for_request(request=request, classname=classname, update_session=True)
    schema = get_schema(classname=classname)
    for iobj in filtered:
        instance = Instance(iobj=iobj, classobj=classobj, classname=classname, schema=schema, expand=True)
        iobj.datapoints = dataview.filter(instance=instance)


    log(DEBUG, aaa=aaa, app="data", view="instance_list", action="get", status="OK", data="{}, {} items".format(classname, len(filtered)))

    # reply
    context["classobj"] = classobj
    context["query"] = query
    context["dataview"] = dataview
    context["paginator"] = paginator
    context["instances"] = filtered
    context["instances_count"] = len(filtered)

    # UI buttons/action
    context["ui_new"] = has_create_permission_on_class(aaa=aaa, classobj=classobj)
    context["ui_export"] = has_export_permission_on_class(aaa=aaa, classobj=classobj)

    return render(request, 'app_data/instance_list.html', context)


# -------------------------------------------------------------------------
# Instance Detail
# -------------------------------------------------------------------------

def instance_detail(request, classname=None, handle=None):
    ''' '''
    context = start_view(request, app="data", view="instance_detail", noauth="app_sirene:index", 
        perm="p_data_read", noauthz="app_data:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]


    classobj = get_class_by_name(classname)
    if not classobj:
        return redirect("app_data:private")
   
    instance = Instance(classobj=classobj, handle=handle, expand=True)
    if not instance.iobj:
        return redirect("app_data:private")
    
    # Check PERMISSIONS
    if not has_read_permission_on_instance(aaa=aaa, iobj=instance.iobj):
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="data", view="instance_detail", action="get", status="KO", data=f"Not allowed on {classname}:{instance.keyname}")
        return redirect("app_data:private")


    # more detailled view ?
    switch_detail = request.GET.get("detail",None)
    want_detail = request.session.get("ui_want_detail", "no")
    if switch_detail:
        if want_detail == "yes":
            want_detail = "no"
        else:
            want_detail = "yes"
        request.session["ui_want_detail"] = want_detail

    if want_detail =="yes":
        form_ui = instance.get_dict_for_ui_detail(skip_external=False, skip_injected=False)
    else:
        form_ui = instance.get_dict_for_ui_detail(skip_external=True, skip_injected=False)


    log(DEBUG, aaa=aaa, app="data", view="instance", action="detail", status="OK", 
        data=f"{classname} / {handle} /  {instance.keyname}")

    context["instance"] = form_ui
    context["classobj"] = classobj
    context["dataclasses"] = get_classes()

    # UI button/action
    context["ui_edit"] = has_edit_permission_on_instance(aaa=aaa, iobj=instance.iobj)
    context["ui_delete"] = has_delete_permission_on_instance(aaa=aaa, iobj=instance.iobj)

    if want_detail=="yes":
        return render(request, 'app_data/instance_detail_tech.html', context)
    else:
        return render(request, 'app_data/instance_detail.html', context)
    


# -------------------------------------------------------------------------
# Instance EDIT
# -------------------------------------------------------------------------

def instance_edit(request, classname=None, handle=None):
    
    context = start_view(request, app="data", view="instance_edit", noauth="app_sirene:index", 
        perm="p_data_update", noauthz="app_data:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    classobj = get_class_by_name(classname)
    if not classobj:
        return redirect("app_data:private")

    instance = Instance(classobj=classobj, handle=handle)
    if not instance.iobj:
        return redirect("app_data:instance_list", classname)

    # Check PERMISSIONS
    if not has_edit_permission_on_instance(aaa=aaa, iobj=instance.iobj):
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="data", view="instance_edit", action="post", status="KO", data=f"Not allowed on {classname}:{instance.keyname}")         
        return redirect("app_data:private")

    if request.method == "POST":

        instance.merge_edit_post_request(request)

        if instance.is_valid():
            r = instance.update()
            if r:
                messages.add_message(request, messages.SUCCESS, _("Instance updated"))
                log(INFO, aaa=aaa, app="data", view="instance_edit", action="post", status="OK", data=f"{classname}:{instance.keyname}") 
                #return redirect("app_data:instance_list", classname)
                return redirect("app_data:instance_detail", classname, instance.handle)
            else:
                messages.add_message(request, messages.ERROR, _("Failed to update"))
                log(ERROR, aaa=aaa, app="data", view="instance_edit", action="post", status="KO", data=f"{classname}:{instance.keyname}") 
        else:
            err = '; '.join(instance.errors)
            messages.add_message(request, messages.ERROR, _("Invalid form") + ': ' + err)
            log(ERROR, aaa=aaa, app="data", view="instance_edit", action="post", status="KO", data=f"{err}") 

        form_ui = instance.get_dict_for_ui_form()

    else:
        form_ui = instance.get_dict_for_ui_form()
        log(DEBUG, aaa=aaa, app="data", view="instance_edit", action="get", status="OK", 
            data=f"{classname} / {instance.keyname}")

    context["classobj"] = classobj
    context["formular"] = form_ui
    context["dataclasses"] = get_classes()
    return render(request, 'app_data/instance_edit.html', context)


# -------------------------------------------------------------------------
# Instance NEW
# -------------------------------------------------------------------------

def instance_new(request, classname=None):
    ''' Blank form for new instance. POST to save'''

    context = start_view(request, app="data", view="instance_new", noauth="app_sirene:index", 
        perm="p_data_create", noauthz="app_data:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

   
    classobj = get_class_by_name(classname)
    if not classobj:
        return redirect("app_data:private")

    # Check PERMISSIONS
    if not has_create_permission_on_class(aaa=aaa, classobj=classobj):
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="data", view="instance_new", action="post", status="KO", data=f"Not alloewd on {classname}")  
        return redirect("app_data:private")

    # new Data Structure
    instance = Instance(classname=classname)
    if not instance:
        return redirect("app_data:instance_list", classname)


    if request.method == "POST":

        instance.merge_new_post_request(request)

        if instance.is_valid():
            r = instance.create()
            if r:
                messages.add_message(request, messages.SUCCESS, _("Instance created"))
                log(INFO, aaa=aaa, app="data", view="instance_new", action="post", status="OK", data=f"{classname}/{instance.keyname}")                 
                return redirect("app_data:instance_list", classname)
            else:
                messages.add_message(request, messages.ERROR, _("Failed to create."))
                log(ERROR, aaa=aaa, app="data", view="instance_new", action="post", status="KO", data=f"{classname}")                 
        else:
            err = '; '.join(instance.errors)
            messages.add_message(request, messages.ERROR, _("Invalid form") + ': ' + err)
            log(ERROR, aaa=aaa, app="data", view="instance_new", action="post", status="KO", data=f"{err}") 

        form_ui = instance.get_dict_for_ui_form()


    else:
        form_ui = instance.get_dict_for_ui_form()
        log(DEBUG, aaa=aaa, app="data", view="instance_new", action="get", status="OK", data=f"{classname}")

    context["classobj"] = classobj
    context["formular"] = form_ui
    context["dataclasses"] = get_classes()
    return render(request, 'app_data/instance_new.html', context)



# -------------------------------------------------------------------------
# Instance DELETE
# -------------------------------------------------------------------------

def instance_delete(request, classname=None, handle=None):
    ''' POST to delete instance by classname and handle'''
    
    context = start_view(request, app="data", view="instance_edit", noauth="app_sirene:index", 
        perm="p_data_delete", noauthz="app_data:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    
    classobj = get_class_by_name(classname)
    if not classobj:
        log(ERROR, aaa=aaa, app="data", view="instance_delete", action="find", status="KO", 
            data=f"class not found {classname}")
        return redirect("app_data:private")

    instance = Instance(classobj=classobj, handle=handle)
    if not instance.iobj:
        log(ERROR, aaa=aaa, app="data", view="instance_delete", action="find", status="KO", 
            data=f"instance not found {classname}:{instance.keyname}")
        return redirect("app_data:instance_list", classname)

    # Check PERMISSIONS
    if not has_delete_permission_on_instance(aaa=aaa, iobj=instance.iobj):
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="data", view="instance_delete", action="post", status="KO", 
            data=f"Not allowed on {classname}:{instance.keyname}")        
        return redirect("app_data:private")


    if request.method == "POST":

        r = instance.delete()
        if r:
            messages.add_message(request, messages.SUCCESS, _("Instance deleted"))
            log(INFO, aaa=aaa, app="data", view="instance_delete", action="post", status="OK", 
                data=f"{classname}:{instance.keyname}")

        else:
            messages.add_message(request, messages.ERROR, _("Failed to delete"))
            log(ERROR, aaa=aaa, app="data", view="instance_delete", action="post", status="KO", 
                data=f"{classname}:{instance.keyname}")
    else:
        messages.add_message(request, messages.ERROR, _("Invalid request"))
        log(WARNING, aaa=aaa, app="data", view="instance_delete", action="get", status="KO", 
            data="invalid request")

    context["dataclasses"] = get_classes()
    return redirect("app_data:instance_list", classname)



# -------------------------------------------------------------------------
# IMPORT
# -------------------------------------------------------------------------

# def data_import(request):
#     ''' '''

#     context = start_view(request, app="data", view="data_import", noauth="app_sirene:index", 
#         perm="p_data_admin", noauthz="app_data:private")
#     if context["redirect"]:
#         return redirect(context["redirect"])
#     aaa = context["aaa"]

#     if request.method == "POST":

#         form = DataUploadForm(request.POST, request.FILES)
#         if form.is_valid():

#             file = request.FILES["file"]

#             if file.multiple_chunks():
#                 messages.add_message(request, messages.ERROR, _("Import failed - file too big"))
#                 log(ERROR, aaa=aaa, app="data", view="import", action="post", status="KO", data=_("Import failed - file too big"))
#                 return redirect("app_data:private")          


#             if file.name.endswith(".csv"):
#                 err = data_import_csv(file)

#             elif file.name.endswith(".json"):
#                 err = data_import_json(file)

#             elif file.name.endswith(".yaml"):
#                 err = data_import_yaml(file)

#             elif file.name.endswith(".yml"):
#                 err = data_import_yaml(file)

#             else:
#                 messages.add_message(request, messages.ERROR, _("Import failed - unknown file type"))
#                 log(ERROR, aaa=aaa, app="data", view="import", action="post", status="KO", data=_("Import failed - unknown file type"))
#                 return redirect("app_data:private")          

#             # OK
#             messages.add_message(request, messages.SUCCESS, _("Import successful") )    
#             log(INFO, aaa=aaa, app="data", view="import", action="post", status="OK", data=_("Import successful"))
#             return redirect("app_data:private")


#     # Failed
#     messages.add_message(request, messages.ERROR, _("Import failed"))
#     log(ERROR, aaa=aaa, app="data", view="import", action="post", status="KO", data=_("Import failed - not a POST"))
    
#     return redirect("app_data:private")


# -----------------------------------------
# import tool
#-----------------------------------------
def data_import(request):

    context = start_view(request, app="data", view="import", 
        noauth="app_sirene:private", perm="p_data_import", noauthz="app_home:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    rawdata = ""

    if request.method == "POST":

        valid = False
        rawdata = request.POST["rawdata"]
        datalist = None

        # Get TextArea
        try:
            datalist = yaml.safe_load(rawdata)
            valid = True
        except Exception as e:
            messages.add_message(request, messages.ERROR, e)

        # File provided ?
        fileform = DataUploadForm(request.POST, request.FILES)
        
        if fileform.is_valid():
            file = request.FILES["file"]
            if file.multiple_chunks():
                messages.add_message(request, messages.ERROR, _("Import failed - file too big"))
                log(ERROR, aaa=aaa, app="data", view="import", action="post", status="KO", data=_("Import failed - file too big"))
                return redirect("app_data:private")          


        #     if file.name.endswith(".csv"):
        #         err = data_import_csv(file)

        #     elif file.name.endswith(".json"):
        #         err = data_import_json(file)

        #     elif file.name.endswith(".yaml"):
        #         err = data_import_yaml(file)

        #     elif file.name.endswith(".yml"):
        #         err = data_import_yaml(file)

            #if file.name.endswith('.csv'):
            #    datalist = load_file_csv(file, pipeline=pipeline)

            if file.name.endswith('.yml') or file.name.endswith('.yaml'):
                datalist = yaml.load(file, Loader=yaml.SafeLoader)

            elif file.name.endswith('.json'):
                datalist = json.load(file) 

            if type(datalist) is not list:
                messages.add_message(request, messages.ERROR, _("Import failed - invalid file"))
                log(ERROR, aaa=aaa, app="data", view="import", action="file", status="KO", 
                    data=f"Content is not a list : {file.name}")
                datalist = []
            else:
                log(INFO, aaa=aaa, app="data", view="import", action="file", status="OK", 
                    data=f"loaded: {file.name}")
                valid = True


        # apply pipeline if any
        if valid:        
            pipeline = request.POST["pipeline"]
            datalist = apply_pipeline(pipeline=pipeline, datalist=datalist)


        # check only
        if request.POST["submit"] == "verify":
            if valid:
                messages.add_message(request, messages.SUCCESS, _("Check ok"))
                log(DEBUG, aaa=aaa, app="data", view="import", action="check", status="OK")
                # no redirect, keep editing
            else:
                # error already displayed
                pass

        # submit
        if request.POST["submit"] == "import":
            # load twice for references
            count = load_broker(datalist, aaa=aaa)
            count = load_broker(datalist, aaa=aaa)
            messages.add_message(request, messages.SUCCESS, _("Import OK"))
            log(DEBUG, aaa=aaa, app="data", view="import", action="import", status="OK",
                data=f"imported: {count} objects")
            return redirect("app_home:private")

    # import may often alter permissions, group, roles
    # it's best to flush for this user after such action, to avoid stale info or logout/login action.
    # request.session.flush()

    context["upload_form"] = DataUploadForm()
    context["rawdata"] = rawdata
    context["pipelines"] = list_pipelines(is_enabled=True)
    return render(request, 'app_data/import.html', context)


# -----------------------------------------
# export tool
#-----------------------------------------
def data_export(request):

    context = start_view(request, app="data", view="export", 
        noauth="app_sirene:private", perm="p_data_export", noauthz="app_home:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    
    aaa = context["aaa"]


    if request.method == "GET":
        # NEXT : check permission (if classname already available)
        # NEXT : provide a form similar to Modal
        context["ui_export"] = True
        return render(request, 'app_data/export.html', context)

    # POST
    # ----

    classname = request.POST.get('classname')
    m = re.compile(r'^[a-zA-Z0-9()_\/\.\s]*$')
    if not m.match(classname):
        redirect("app_data:private")

    classobj = get_class_by_name(classname)
    if not classobj:
        redirect("app_data:private")

    # Check PERMISSIONS
    if not has_export_permission_on_class(aaa=aaa, classobj=classobj):
        messages.add_message(request, messages.ERROR, _("Not allowed"))
        log(ERROR, aaa=aaa, app="data", view="export", action="post", status="KO", data=f"Not allowed on {classname}")        
        return redirect("app_data:private")


    # max size 
    # NEXT : move to async if too big
    max_size = int(get_configuration("data","EXPORT_INTERACTIVE_MAX_SIZE"))

    # page : thispage / allpages
    page = request.POST.get('page')
    if page not in ["allpages", "thispage"]:
        return redirect("app_data:private")
    if page == "thispage":
        paginator = paginator_for_request(request=request, classobj=classobj, max_size = max_size, update_session=False)
        if not paginator:
            return redirect("app_data:private")
        offset = paginator.offset
        limit = paginator.limit
    else:
        offset = 0
        limit = max_size

    # query
    query = get_query_from_request(request)

    # dataview : name of session dataview or ___ALL___ (no dataview, all cols)
    dv = request.POST.get('dataview', None)
    if dv == "___ALL___":
        dataview = None
    else:
        dataview = dataview_for_request(request=request, classname=classname, update_session=False)

    # csv/yaml/json
    format =  request.POST.get('format')
    if format not in ["csv","yaml","json"]:
        return redirect("app_data:private")

    # instances
    raw_instances = get_instance_from_query(classobj=classobj, query=query, offset=offset, limit=limit)


    # apply permissions
    db_instances = []
    for i in raw_instances:
        if has_read_permission_on_instance(aaa=aaa, iobj=i):
            db_instances.append(i)

    count = len(db_instances)

    # warn if max size reached
    if count == max_size:
        log(WARNING, aaa=aaa, app="data", view="export", action="post", status="KO", data=f"Export may be truncated for {classname} - {max_size}") 
        messages.add_message(request, messages.WARNING, _("Export may be truncated (size max reached)"))



    # if dataview:
    #     schema = get_schema(classname=classname)
    #     for iobj in db_instances:
    #         instance = Instance(iobj=iobj, classobj=classobj, classname=classname, schema=schema, expand=True)
    #         iobj.datapoints = dataview.filter(instance=instance)

    if format == "yaml":
        datalist = []
        schema = get_schema(classname=classname)
        for iobj in db_instances:
            instance = Instance(classname=classname, classobj=classobj, iobj=iobj,  schema=schema, expand=True)
            data = instance.get_dict_for_export()            
            datalist.append(data)
        filedata = yaml.dump(datalist, allow_unicode=True, Dumper=MyYamlDumper, sort_keys=False)
        response = HttpResponse(filedata, content_type='text/yaml')  
        response['Content-Disposition'] = 'attachment; filename="data.yaml"'
        return response

    if format == "json":
        datalist = []
        schema = get_schema(classname=classname)
        for iobj in db_instances:
            instance = Instance(classname=classname, classobj=classobj, iobj=iobj,  schema=schema, expand=True)
            data = instance.get_dict_for_export()            
            datalist.append(data)
        filedata = json.dumps(datalist, indent=4, ensure_ascii=False)
        response = HttpResponse(filedata, content_type='text/json')  
        response['Content-Disposition'] = 'attachment; filename="data.json"'
        return response


    #     #export_data(classes=["app"])
    #     if output == "yaml":
    #         log(INFO, aaa=aaa, app="data", view="instance_list", action="export", status="OK", data="yaml") 
    #         response = data_yaml_response(classes = [classname])
    #         return response 
        
    #     elif output == "json":
    #         log(INFO, aaa=aaa, app="data", view="instance_list", action="export", status="OK", data="json") 
    #         response = data_json_response(classes = [classname])
    #         return response   


    # for i in instances:
    #     #print(i)
    #     pass

    context["ui_export"] = True
    return render(request, 'app_data/export.html', context)




#YAML
class MyYamlDumper(yaml.SafeDumper):
    def write_line_break(self, data=None):
        super().write_line_break(data)
        if len(self.indents) < 2:
            super().write_line_break()
